/*
 * Copyright (c) 1997, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2016, IS2T - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.security;

import java.io.IOException;
import java.io.InputStream;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.Enumeration;

/**
 * This class represents a storage facility for cryptographic
 * keys and certificates.
 *
 * <p> A {@code KeyStore} manages different types of entries.
 * Each type of entry implements the {@code KeyStore.Entry} interface.
 * Three basic {@code KeyStore.Entry} implementations are provided:
 *
 * <ul>
 * <li><b>KeyStore.PrivateKeyEntry</b>
 * <p> This type of entry holds a cryptographic {@code PrivateKey},
 * which is optionally stored in a protected format to prevent
 * unauthorized access.  It is also accompanied by a certificate chain
 * for the corresponding public key.
 *
 * <p> Private keys and certificate chains are used by a given entity for
 * self-authentication. Applications for this authentication include software
 * distribution organizations which sign JAR files as part of releasing
 * and/or licensing software.
 *
 * <li><b>KeyStore.SecretKeyEntry</b>
 * <p> This type of entry holds a cryptographic {@code SecretKey},
 * which is optionally stored in a protected format to prevent
 * unauthorized access.
 *
 * <li><b>KeyStore.TrustedCertificateEntry</b>
 * <p> This type of entry contains a single public key {@code Certificate}
 * belonging to another party. It is called a <i>trusted certificate</i>
 * because the keystore owner trusts that the public key in the certificate
 * indeed belongs to the identity identified by the <i>subject</i> (owner)
 * of the certificate.
 *
 * <p>This type of entry can be used to authenticate other parties.
 * </ul>
 *
 * <p> Each entry in a keystore is identified by an "alias" string. In the
 * case of private keys and their associated certificate chains, these strings
 * distinguish among the different ways in which the entity may authenticate
 * itself. For example, the entity may authenticate itself using different
 * certificate authorities, or using different public key algorithms.
 *
 * <p> Whether aliases are case sensitive is implementation dependent. In order
 * to avoid problems, it is recommended not to use aliases in a KeyStore that
 * only differ in case.
 *
 * <p> Whether keystores are persistent, and the mechanisms used by the
 * keystore if it is persistent, are not specified here. This allows
 * use of a variety of techniques for protecting sensitive (e.g., private or
 * secret) keys. Smart cards or other integrated cryptographic engines
 * (SafeKeyper) are one option, and simpler mechanisms such as files may also
 * be used (in a variety of formats).
 *
 * <p> Typical ways to request a KeyStore object include
 * relying on the default type and providing a specific keystore type.
 *
 * <ul>
 * <li>To rely on the default type:
 * <pre>
 *    KeyStore ks = KeyStore.getInstance(KeyStore.getDefaultType());
 * </pre>
 * The system will return a keystore implementation for the default type.
 *
 * <li>To provide a specific keystore type:
 * <pre>
 *      KeyStore ks = KeyStore.getInstance("JKS");
 * </pre>
 * The system will return the most preferred implementation of the
 * specified keystore type available in the environment. <p>
 * </ul>
 *
 * <p> Before a keystore can be accessed, it must be
 * {@link #load(java.io.InputStream, char[]) loaded}.
 * <pre>
 *    KeyStore ks = KeyStore.getInstance(KeyStore.getDefaultType());
 *
 *    // get user password and file input stream
 *    char[] password = getPassword();
 *
 *    try (FileInputStream fis = new FileInputStream("keyStoreName")) {
 *        ks.load(fis, password);
 *    }
 * </pre>
 *
 * To create an empty keystore using the above {@code load} method,
 * pass {@code null} as the {@code InputStream} argument.
 *
 * <p> Once the keystore has been loaded, it is possible
 * to read existing entries from the keystore, or to write new entries
 * into the keystore:
 * <pre>
 *    KeyStore.ProtectionParameter protParam =
 *        new KeyStore.PasswordProtection(password);
 *
 *    // get my private key
 *    KeyStore.PrivateKeyEntry pkEntry = (KeyStore.PrivateKeyEntry)
 *        ks.getEntry("privateKeyAlias", protParam);
 *    PrivateKey myPrivateKey = pkEntry.getPrivateKey();
 *
 *    // save my secret key
 *    javax.crypto.SecretKey mySecretKey;
 *    KeyStore.SecretKeyEntry skEntry =
 *        new KeyStore.SecretKeyEntry(mySecretKey);
 *    ks.setEntry("secretKeyAlias", skEntry, protParam);
 *
 *    // store away the keystore
 *    try (FileOutputStream fos = new FileOutputStream("newKeyStoreName")) {
 *        ks.store(fos, password);
 *    }
 * </pre>
 *
 * Note that although the same password may be used to
 * load the keystore, to protect the private key entry,
 * to protect the secret key entry, and to store the keystore
 * (as is shown in the sample code above),
 * different passwords or other protection parameters
 * may also be used.
 *
 * <p> Every implementation of the Java platform is required to support
 * the following standard {@code KeyStore} type:
 * <ul>
 * <li>{@code PKCS12}</li>
 * </ul>
 * This type is described in the <a href=
 * "{@docRoot}/../technotes/guides/security/StandardNames.html#KeyStore">
 * KeyStore section</a> of the
 * Java Cryptography Architecture Standard Algorithm Name Documentation.
 * Consult the release documentation for your implementation to see if any
 * other types are supported.
 *
 * @author Jan Luehe
 *
 * @see java.security.cert.Certificate
 *
 * @since 1.2
 */

public class KeyStore {



	/**
	 * Returns a keystore object of the specified type.
	 *
	 * <p> This method traverses the list of registered security Providers,
	 * starting with the most preferred Provider.
	 * A new KeyStore object encapsulating the
	 * KeyStoreSpi implementation from the first
	 * Provider that supports the specified type is returned.
	 *
	 *
	 * @param type the type of keystore.
	 * See the KeyStore section in the <a href=
	 * "{@docRoot}/../technotes/guides/security/StandardNames.html#KeyStore">
	 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
	 * for information about standard keystore types.
	 *
	 * @return a keystore object of the specified type.
	 *
	 * @exception KeyStoreException if no Provider supports a
	 *          KeyStoreSpi implementation for the
	 *          specified type.
	 *
	 */
	public static KeyStore getInstance(String type)
			throws KeyStoreException
	{
		throw new RuntimeException();
	}



	/**
	 * Returns the default keystore type as specified by the
	 * {@code keystore.type} security property, or the string
	 * {@literal "jks"} (acronym for {@literal "Java keystore"})
	 * if no such property exists.
	 *
	 * <p>The default keystore type can be used by applications that do not
	 * want to use a hard-coded keystore type when calling one of the
	 * {@code getInstance} methods, and want to provide a default keystore
	 * type in case a user does not specify its own.
	 *
	 * <p>The default keystore type can be changed by setting the value of the
	 * {@code keystore.type} security property to the desired keystore type.
	 *
	 * @return the default keystore type as specified by the
	 * {@code keystore.type} security property, or the string {@literal "jks"}
	 * if no such property exists.
	 */
	public final static String getDefaultType() {
		throw new RuntimeException();
	}


	/**
	 * Returns the type of this keystore.
	 *
	 * @return the type of this keystore.
	 */
	public final String getType()
	{
		throw new RuntimeException();
	}


	/**
	 * Returns the certificate associated with the given alias.
	 *
	 * <p> If the given alias name identifies an entry
	 * created by a call to {@code setCertificateEntry},
	 * or created by a call to {@code setEntry} with a
	 * {@code TrustedCertificateEntry},
	 * then the trusted certificate contained in that entry is returned.
	 *
	 * <p> If the given alias name identifies an entry
	 * created by a call to {@code setKeyEntry},
	 * or created by a call to {@code setEntry} with a
	 * {@code PrivateKeyEntry},
	 * then the first element of the certificate chain in that entry
	 * is returned.
	 *
	 * @param alias the alias name
	 *
	 * @return the certificate, or null if the given alias does not exist or
	 * does not contain a certificate.
	 *
	 * @exception KeyStoreException if the keystore has not been initialized
	 * (loaded).
	 */
	public final Certificate getCertificate(String alias)
			throws KeyStoreException
	{
		throw new RuntimeException();
	}



	/**
	 * Assigns the given trusted certificate to the given alias.
	 *
	 * <p> If the given alias identifies an existing entry
	 * created by a call to {@code setCertificateEntry},
	 * or created by a call to {@code setEntry} with a
	 * {@code TrustedCertificateEntry},
	 * the trusted certificate in the existing entry
	 * is overridden by the given certificate.
	 *
	 * @param alias the alias name
	 * @param cert the certificate
	 *
	 * @exception KeyStoreException if the keystore has not been initialized,
	 * or the given alias already exists and does not identify an
	 * entry containing a trusted certificate,
	 * or this operation fails for some other reason.
	 */
	public final void setCertificateEntry(String alias, Certificate cert)
			throws KeyStoreException
	{
		throw new RuntimeException();
	}

	/**
	 * Deletes the entry identified by the given alias from this keystore.
	 *
	 * @param alias the alias name
	 *
	 * @exception KeyStoreException if the keystore has not been initialized,
	 * or if the entry cannot be removed.
	 */
	public final void deleteEntry(String alias)
			throws KeyStoreException
	{
		throw new RuntimeException();
	}

	/**
	 * Lists all the alias names of this keystore.
	 *
	 * @return enumeration of the alias names
	 *
	 * @exception KeyStoreException if the keystore has not been initialized
	 * (loaded).
	 */
	public final Enumeration<String> aliases()
			throws KeyStoreException
	{
		throw new RuntimeException();
	}

	/**
	 * Checks if the given alias exists in this keystore.
	 *
	 * @param alias the alias name
	 *
	 * @return true if the alias exists, false otherwise
	 *
	 * @exception KeyStoreException if the keystore has not been initialized
	 * (loaded).
	 */
	public final boolean containsAlias(String alias)
			throws KeyStoreException
	{
		throw new RuntimeException();
	}

	/**
	 * Retrieves the number of entries in this keystore.
	 *
	 * @return the number of entries in this keystore
	 *
	 * @exception KeyStoreException if the keystore has not been initialized
	 * (loaded).
	 */
	public final int size()
			throws KeyStoreException
	{
		throw new RuntimeException();
	}


	/**
	 * Returns true if the entry identified by the given alias
	 * was created by a call to {@code setCertificateEntry},
	 * or created by a call to {@code setEntry} with a
	 * {@code TrustedCertificateEntry}.
	 *
	 * @param alias the alias for the keystore entry to be checked
	 *
	 * @return true if the entry identified by the given alias contains a
	 * trusted certificate, false otherwise.
	 *
	 * @exception KeyStoreException if the keystore has not been initialized
	 * (loaded).
	 */
	public final boolean isCertificateEntry(String alias)
			throws KeyStoreException
	{
		throw new RuntimeException();
	}

	/**
	 * Loads this KeyStore from the given input stream.
	 *
	 * <p>A password may be given to unlock the keystore
	 * (e.g. the keystore resides on a hardware token device),
	 * or to check the integrity of the keystore data.
	 * If a password is not given for integrity checking,
	 * then integrity checking is not performed.
	 *
	 * <p>In order to create an empty keystore, or if the keystore cannot
	 * be initialized from a stream, pass {@code null}
	 * as the {@code stream} argument.
	 *
	 * <p> Note that if this keystore has already been loaded, it is
	 * reinitialized and loaded again from the given input stream.
	 *
	 * @param stream the input stream from which the keystore is loaded,
	 * or {@code null}
	 * @param password the password used to check the integrity of
	 * the keystore, the password used to unlock the keystore,
	 * or {@code null}
	 *
	 * @exception IOException if there is an I/O or format problem with the
	 * keystore data, if a password is required but not given,
	 * or if the given password was incorrect. If the error is due to a
	 * wrong password, the {@link Throwable#getCause cause} of the
	 * {@code IOException} should be an
	 * {@code UnrecoverableKeyException}
	 * @exception NoSuchAlgorithmException if the algorithm used to check
	 * the integrity of the keystore cannot be found
	 * @exception CertificateException if any of the certificates in the
	 * keystore could not be loaded
	 */
	public final void load(InputStream stream, char[] password)
			throws IOException, NoSuchAlgorithmException, CertificateException
	{
		throw new RuntimeException();
	}


}
