/*
 * Copyright 2016-2022 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.animation;

import ej.basictool.ArrayTools;
import ej.basictool.ThreadUtils;
import ej.bon.Util;
import ej.microui.MicroUI;
import ej.microui.display.Display;

/**
 * An animator executes animations as fast as possible.
 * <p>
 * The animator loops on:
 * <ul>
 * <li>retrieve the current time with {@link ej.bon.Util#platformTimeMillis()},</li>
 * <li>call the {@link Animation#tick(long)} of all the registered animations,</li>
 * <li>wait for the end of the flush,</li>
 * <li>start again while at least one animation is running.</li>
 * </ul>
 * <p>
 * Only one animator can be running at any time.
 */
public class Animator {

	private final Object monitor;
	private Animation[] animations;
	private boolean started;
	private boolean ticking;

	/**
	 * Creates an animator.
	 * <p>
	 * Multiple animators may be created but only one can be running at any time.
	 */
	public Animator() {
		this.monitor = new Object();
		this.animations = new Animation[0];
		this.started = false;
		this.ticking = false;
	}

	/**
	 * Starts an animation.
	 * <p>
	 * The animation tick method is then called every period.
	 * <p>
	 * If the animation is already started, nothing is done.
	 *
	 * @param animation
	 *            the animation to start.
	 * @see Animation#tick(long)
	 */
	public void startAnimation(Animation animation) {
		synchronized (this.monitor) {
			Animation[] animations = this.animations;
			if (!ArrayTools.contains(animations, animation)) {
				this.animations = ArrayTools.add(animations, animation);

				if (!this.started) {
					this.started = true;
					callOnFlushCompleted(createTickRunnable());
				}
			}
		}
	}

	/**
	 * Stops an animation.
	 * <p>
	 * The tick method of the given animation will not be called anymore by this animator.
	 * <p>
	 * This method must be called in the MicroUI thread.
	 *
	 * @param animation
	 *            the animation to stop.
	 * @throws IllegalStateException
	 *             if this method is called in an other thread than the MicroUI thread or during an animation tick.
	 */
	public void stopAnimation(Animation animation) {
		if (!MicroUI.isUIThread() || this.ticking) {
			throw new IllegalStateException();
		}

		stopAnimationNoCheck(animation);
	}

	/**
	 * Stops all the animations.
	 * <p>
	 * The tick method of every animation will not be called anymore by this animator.
	 * <p>
	 * This method must be called in the MicroUI thread.
	 *
	 * @throws IllegalStateException
	 *             if this method is called in an other thread than the MicroUI thread or during an animation tick.
	 */
	public void stopAllAnimations() {
		if (!MicroUI.isUIThread() || this.ticking) {
			throw new IllegalStateException();
		}

		synchronized (this.monitor) {
			if (this.animations.length > 0) {
				this.animations = new Animation[0];
				this.started = false;
				callOnFlushCompleted(createEmptyRunnable());
			}
		}
	}

	private void stopAnimationNoCheck(Animation animation) {
		synchronized (this.monitor) {
			Animation[] newAnimations = ArrayTools.remove(this.animations, animation);
			if (newAnimations != this.animations) {
				this.animations = newAnimations;

				if (newAnimations.length == 0) {
					this.started = false;
					callOnFlushCompleted(createEmptyRunnable());
				}
			}
		}
	}

	private void tick() {
		long currentTimeMillis = Util.platformTimeMillis();
		// Get the array in a locale to avoid synchronization.
		Animation[] animations = this.animations;
		for (Animation animation : animations) {
			assert animation != null;
			try {
				if (!animation.tick(currentTimeMillis)) {
					stopAnimationNoCheck(animation);
				}
			} catch (Exception e) {
				ThreadUtils.handleUncaughtException(e);
				stopAnimationNoCheck(animation);
			}
		}
	}

	private void runTick(Runnable runnable) {
		this.ticking = true;
		tick();
		this.ticking = false;

		if (this.started) {
			callOnFlushCompleted(runnable);
		}
	}

	private void callOnFlushCompleted(Runnable runnable) {
		Display display = Display.getDisplay();
		display.callOnFlushCompleted(runnable);
		// Make sure that the callOnFlushCompleted will be executed.
		display.requestFlush();
	}

	private Runnable createTickRunnable() {
		return new Runnable() {
			@Override
			public void run() {
				runTick(this);
			}
		};
	}

	// M0092MEJAUI-2290: Display.callOnFlushCompleted() can not be called with null parameter for the moment
	private static Runnable createEmptyRunnable() {
		return new Runnable() {
			@Override
			public void run() {
				// do nothing
			}
		};
	}
}
