/*
 * Copyright 2016-2019 MicroEJ Corp. - EDC compliance and optimizations.
 * Use is subject to license terms.
 */
package org.junit;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

//<edited for EDC compliance>
/**
 * The <code>Test</code> annotation tells JUnit that the <code>public void</code> method to which it is attached can be
 * run as a test case. To run the method, JUnit first constructs a fresh instance of the class then invokes the
 * annotated method. Any exceptions thrown by the test will be reported by JUnit as a failure. If no exceptions are
 * thrown, the test is assumed to have succeeded.
 * <p>
 * A simple test looks like this:
 *
 * <pre>
 * public class Example {
 *    <b>&#064;Test</b>
 *    public void method() {
 *       org.junit.Assert.assertTrue( new ArrayList().isEmpty() );
 *    }
 * }
 * </pre>
 * <p>
 * The <code>Test</code> annotation supports one optional parameter. This parameter, <code>expected</code>, declares
 * that a test method should throw an exception. If it doesn't throw an exception or if it throws a different exception
 * than the one declared, the test fails. For example, the following test succeeds:
 *
 * <pre>
 *    &#064;Test(<b>expected=IndexOutOfBoundsException.class</b>) public void outOfBounds() {
 *       new ArrayList&lt;Object&gt;().get(1);
 *    }
 * </pre>
 *
 * @since 4.0
 */
@Retention(RetentionPolicy.RUNTIME)
@Target({ ElementType.METHOD })
public @interface Test {

	/**
	 * Default empty exception
	 */
	static class None extends Throwable {
		private static final long serialVersionUID = 1L;

		private None() {
		}
	}

	/**
	 * Optionally specify <code>expected</code>, a Throwable, to cause a test method to succeed if and only if an
	 * exception of the specified class is thrown by the method.
	 */
	Class<? extends Throwable> expected() default None.class;

	// <removed for EDC compliance> /**
	// <removed for EDC compliance> * Optionally specify <code>timeout</code> in milliseconds to cause a test method to
	// fail if it
	// <removed for EDC compliance> * takes longer than that number of milliseconds.
	// <removed for EDC compliance> * <p>
	// <removed for EDC compliance> * <b>THREAD SAFETY WARNING:</b> Test methods with a timeout parameter are run in a
	// thread other than the
	// <removed for EDC compliance> * thread which runs the fixture's @Before and @After methods. This may yield
	// different behavior for
	// <removed for EDC compliance> * code that is not thread safe when compared to the same test method without a
	// timeout parameter.
	// <removed for EDC compliance> * <b>Consider using the {@link org.junit.rules.Timeout} rule instead</b>, which
	// ensures a test method is run on the
	// <removed for EDC compliance> * same thread as the fixture's @Before and @After methods.
	// <removed for EDC compliance> * </p>
	// <removed for EDC compliance> */
	// <removed for EDC compliance> long timeout() default 0L;
}
