/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display.buffer;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.Widget;
import ej.fp.widget.display.DisplayBufferManager;
import ej.microui.display.Rectangle;

/**
 * Two buffers in RAM and one buffer on the display.
 */
public class TransmitSwapBufferPolicy implements DisplayBufferPolicy {

	private Widget displayWidget;
	private int displayWidth;
	private int displayHeight;

	private Image bufferA;
	private Image bufferB;
	private Image frontBuffer;
	private boolean drawingOnBufferA;

	private int currentFlush;
	private int ackFlush;

	@Override
	public void setDisplayProperties(Widget displayWidget, int width, int height, int initialColor) {
		this.displayWidget = displayWidget;
		this.displayWidth = width;
		this.displayHeight = height;
		this.bufferA = FrontPanel.getFrontPanel().newImage(this.displayWidth, this.displayHeight, initialColor, false);
		this.bufferB = FrontPanel.getFrontPanel().newImage(this.displayWidth, this.displayHeight, initialColor, false);
		this.frontBuffer = FrontPanel.getFrontPanel().newImage(this.displayWidth, this.displayHeight, initialColor,
				false);
	}

	@Override
	public int getBufferCount() {
		return 2;
	}

	@Override
	public boolean isDoubleBuffered() {
		return true;
	}

	@Override
	public void flush(DisplayBufferManager bufferManager, Rectangle[] rectangles) {
		// Ensure that previous flush is finished.
		Object monitor = bufferManager.getMonitor();
		synchronized (monitor) {
			while (this.currentFlush != this.ackFlush) {
				try {
					monitor.wait();
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					break;
				}
			}
		}
		// Transmit what has changed to the display.
		for (Rectangle rectangle : rectangles) {
			this.frontBuffer.drawImage(getBackBuffer(), rectangle.getX1(), rectangle.getY1(), rectangle.getWidth(),
					rectangle.getHeight(), rectangle.getX1(), rectangle.getY1(), rectangle.getWidth(),
					rectangle.getHeight());
		}
		swap();
		this.currentFlush++;
		this.displayWidget.repaint();
		new Thread() {
			@Override
			public void run() {
				bufferManager.simulateFlushTime();
				tryToReuse(bufferManager);
			}

		}.start();
	}

	private void swap() {
		this.drawingOnBufferA = !this.drawingOnBufferA;
	}

	private void tryToReuse(DisplayBufferManager bufferManager) {
		Object monitor = bufferManager.getMonitor();
		synchronized (monitor) {
			if (bufferManager.isCleanBuffer()) {
				swap();
			}
			this.ackFlush++;
			monitor.notifyAll();
		}
	}

	@Override
	public Image getBackBuffer() {
		if (this.drawingOnBufferA) {
			return this.bufferA;
		} else {
			return this.bufferB;
		}
	}

	@Override
	public Image getFrontBuffer() {
		return this.frontBuffer;
	}

	@Override
	public void dispose() {
		FrontPanel.getFrontPanel().disposeIfNotNull(this.bufferA, this.bufferB);
	}

}
