/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display.buffer;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.Widget;
import ej.fp.widget.display.DisplayBufferManager;
import ej.microui.display.Rectangle;

/**
 * Several buffers in RAM used alternatively for the display.
 */
public abstract class SwapBufferPolicy implements DisplayBufferPolicy {

	private final int bufferCount;
	private Widget displayWidget;

	private Image[] buffers;
	private int backBuffer;

	/**
	 * Creates a swap buffer policy.
	 *
	 * @param bufferCount
	 *            the number of buffers.
	 */
	public SwapBufferPolicy(int bufferCount) {
		this.bufferCount = bufferCount;
		if (bufferCount <= 1) {
			throw new IllegalArgumentException("This buffer manager can only handle at least 2 buffers."); //$NON-NLS-1$
		}
	}

	@Override
	public void setDisplayProperties(Widget displayWidget, int width, int height, int initialColor) {
		this.displayWidget = displayWidget;
		this.buffers = new Image[this.bufferCount];
		for (int i = 0; i < this.bufferCount; i++) {
			this.buffers[i] = FrontPanel.getFrontPanel().newImage(width, height, initialColor, false);
		}
	}

	@Override
	public int getBufferCount() {
		return this.buffers.length;
	}

	@Override
	public boolean isDoubleBuffered() {
		return true;
	}

	@Override
	public void flush(DisplayBufferManager bufferManager, Rectangle[] rectangles) {
		swap();
		this.displayWidget.repaint();
		bufferManager.simulateFlushTime();
	}

	private void swap() {
		this.backBuffer = (this.backBuffer + 1) % (this.buffers.length);
	}

	@Override
	public Image getBackBuffer() {
		return this.buffers[this.backBuffer];
	}

	@Override
	public Image getFrontBuffer() {
		return this.buffers[getPreviousBuffer()];
	}

	private int getPreviousBuffer() {
		int count = this.buffers.length;
		return (this.backBuffer - 1 + count) % (count);
	}

	@Override
	public void dispose() {
		FrontPanel.getFrontPanel().disposeIfNotNull(this.buffers);
	}

}
