/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display.brs;

import ej.fp.widget.display.DisplayBufferManager;
import ej.microui.display.Rectangle;

/**
 * The "single" strategy is dedicated to the display with one buffer on the MCU side and one buffer on the display side
 * (the display front buffer is not mapped to the MCU address space). The refresh consists in transmitting the back
 * buffer data to the display front buffer. The back buffer always contains the past (all the drawings since the
 * startup) and by consequence nothing needs to be restored.
 */
public class SingleRefreshStrategy implements BufferRefreshStrategy {

	private int displayWidth;
	private int displayHeight;

	@Override
	public void setDisplaySize(int width, int height) {
		this.displayWidth = width;
		this.displayHeight = height;
	}

	@Override
	public void setBufferCount(int bufferCount) {
		if (bufferCount != 1) {
			throw new IllegalArgumentException(
					"This refresh strategy supports only one buffer, check the buffer policy selected in the frontpanel configuration file."); //$NON-NLS-1$
		}
	}

	@Override
	public void newDrawingRegion(DisplayBufferManager bufferManager, Rectangle rectangle, boolean drawingNow) {
		// Nothing to do.
	}

	@Override
	public void refresh(DisplayBufferManager bufferManager) {
		bufferManager.flush(new Rectangle[] { new Rectangle(0, 0, this.displayWidth, this.displayHeight) });
	}

}
