/*
 * Java
 *
 * Copyright 2008-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp;

import java.util.ArrayList;
import java.util.List;

/**
 * A device represents the physical look of a device.
 * <p>
 * It contains all of the input/output parts of the real device (screens, buttons, ...). The packaging of the device is
 * represented by a background image.
 * <p>
 * A device is created by the front panel framework during the parsing step. As soon as all device's widgets are created
 * and finalized, the device is started. A this moment the widgets and user code can call the method
 * {@link Device#getDevice()} (and {@link FrontPanel#getDevice()}): it will return the unique device. Before starting
 * step, a call to these functions throws a runtime exception.
 */
public abstract class Device extends Composite {

	// queue of input events
	private final EventQueue eventGenerator;

	/**
	 * Creates a new device with its name (label) and background.
	 *
	 * @param name
	 *            the name to show as title.
	 * @param skin
	 *            the device background image.
	 */
	public Device(String name, Image skin) {
		setSkin(skin);
		setLabel(name);
		this.eventGenerator = new EventQueue();
	}

	/**
	 * Gets the unique device instance.
	 * <p>
	 * Equivalent to <code>FrontPanel.getFrontPanel().getDevice()</code>.
	 *
	 * @return the unique device instance.
	 */
	public static Device getDevice() {
		return FrontPanel.getFrontPanel().getDevice();
	}

	/**
	 * Gets the name of the device.
	 * <p>
	 * Equivalent to <code>getLabel()</code>.
	 *
	 * @return the device's name.
	 */
	public String getName() {
		return getLabel();
	}

	/**
	 * Gets a list of device's widgets.
	 *
	 * @return all device's widgets in a list.
	 */
	public List<Widget> getWidgets() {
		List<Widget> widgets = new ArrayList<>(0);
		for (Widget widget : this) {
			widgets.add(widget);
		}
		return widgets;
	}

	/**
	 * Gets a list of device's widgets which implement the given class.
	 *
	 * @param clazz
	 *            the widget class to retrieve.
	 * @return device's widgets in a list.
	 */
	public <T> List<T> getWidgets(Class<T> clazz) {
		List<T> widgets = new ArrayList<>(0);
		for (Widget widget : this) {
			if (clazz.isInstance(widget)) {
				T t = clazz.cast(widget);
				widgets.add(t);
			}
		}
		return widgets;
	}

	/**
	 * Gets the first width which implement the given class.
	 *
	 * @param clazz
	 *            the widget class to retrieve.
	 * @return the expected widget or null if not widget respect the given conditions.
	 */
	public <T> T getWidget(Class<T> clazz) {
		return getWidget(clazz, null);
	}

	/**
	 * Gets the first width which implement the given class and whose label is equals to given label.
	 *
	 * @param clazz
	 *            the widget class to retrieve.
	 * @param label
	 *            the label to looking for or null to return only the first widget which implements the given class.
	 * @return the expected widget or null if not widget respect the given conditions.
	 */
	public <T> T getWidget(Class<T> clazz, String label) {
		for (Widget widget : this) {
			if (clazz.isInstance(widget) && (label == null || label.equals(widget.getLabel()))) {
				return clazz.cast(widget);
			}
		}
		return null;
	}

	/**
	 * Gets the input events queue.
	 *
	 * @return the input events queue.
	 */
	public EventQueue getEventQueue() {
		return this.eventGenerator;
	}

}
