/*
 * Java
 *
 * Copyright 2014-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.websocket;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.Socket;

import ej.util.message.Level;
import ej.websocket.frame.FrameScanner;
import ej.websocket.frame.RawFrame;

/**
 * When a websocket connection is established (that is, when the handskahe is done), the {@link WebSocket} will start a new thread running a
 * {@link Receiver} runnable. The {@link Receiver} is responsible for managing incoming data.
 * <p>
 * It reads the input stream, scanning for frames thanks to a {@link FrameScanner}. When a frame is completely received by this {@link FrameScanner},
 * the {@link Receiver} checks for its correctness and choose how to react to it. It answers to control frames such as ping or close and use the
 * {@link WebSocket}'s {@link Endpoint} for high level events. If an error occurs it calls {@link Endpoint#onError(WebSocket, Throwable)}.
 *
 *
 */
/* default */class Receiver implements Runnable {

	private final WebSocket ws;
	private final FrameScanner scanner;

	/**
	 * @param ws
	 *            the associated {@link WebSocket}
	 */
	/* default */ Receiver(WebSocket ws) {
		this.ws = ws;
		this.scanner = new FrameScanner(ws.getInputStream());
	}

	@Override
	public void run() {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.RECEIVER_START, this);

		try {
			RawFrame fullFrame = null;
			while (ws.getCurrentState() != ConnectionStates.CLOSED) {
				boolean isComplete = true;
				// Get a new frame
				RawFrame f = scanner.getFrame();
				Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.RECEIVER_FRAME_RECEIVED, this);

				// A fragmented message is received
				if(f.getFIN() == 0) {
					// It is a begining of a fragmented message
					if (fullFrame == null && f.getOpcode() != RawFrame.OPCODE_CONTINUATION_FRAME) {
						fullFrame = f;
						isComplete = false;
					}
					// Some already arrived.
					else if (fullFrame != null && f.getOpcode() == RawFrame.OPCODE_CONTINUATION_FRAME) {
						fullFrame.append(f);
						isComplete = false;
					}
					// The end of a fragmented message arrived.
				} else if (f.getFIN() == 1 && fullFrame != null) {
					if(f.getOpcode()==0) {
						fullFrame.append(f);
						f = fullFrame;
						fullFrame = null;
					} else if (f.getOpcode() != RawFrame.OPCODE_PING && f.getOpcode() != RawFrame.OPCODE_PONG
							&& f.getOpcode() != RawFrame.OPCODE_CONNECTION_CLOSE) {
						// Invalid sequence.
						throw new IOException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
								Messages.INVALID_SEQUENCE));
					}
				}

				// Debugging
				// new PrettyFramePrinter(System.out).print(f);
				if (isComplete) {
					handleFrame(f);
				}
			}
		} catch (Exception e) {
			handleError(e);
		}
		/*
		 * If we get here and no exception occurred, it means that the connection has been closed. We don't have to call onClose() on the endpoint
		 * since the connection is already closed (and consequently onClose() has been called already.
		 */
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.RECEIVER_TERMINATED, this);
	}

	private void handleError(Exception e) {
		if (ws.getCurrentState() == ConnectionStates.OPEN) {
			/*
			 * If we get here, it is very likely that the TCP connection has been lost unexpectedly. Note that this would normally not happen when
			 * the websocket is closed normally. Indeed, if local application starts the closing handshake, the websocket moves to the CLOSING
			 * state and a response closing frame is received (by getFrame()) and properly managed (by handle()) without generating IOException.
			 */
			Messages.LOGGER.log(Level.SEVERE, Messages.CATEGORY, Messages.ERROR_UNKNOWN, e, this);
			WebSocketException wse = new WebSocketException(e);
			ws.getEndpoint().onError(ws, wse);
			ws.getEndpoint().onClose(ws, new ReasonForClosure(CloseCodes.CONNECTION_CLOSED_ABNORMALLY, null));

		} else {
			/*
			 * This case happens when the OnTimeOutCloser closes the underlying TCP connection. Hence, the websocket state is CLOSED and the TCP
			 * connection is closed so an IOException occurs in getFrame() because its calls read() on an invalid input stream. There is nothing
			 * to do since both websocket and TCP socket (and associated streams) are closed.
			 */
			/*
			 * This case happens also when the closing handshaking has been started (to the connection is in the CLOSING state) and the peer
			 * brutally drops the TCP connection (Wireshark shows [RST,ACK] packets).
			 */
			Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.TIMEOUT, e, this);
		}

		// It is probably already closed but we want to free the resources for sure
		ws.closeUnderlyingTCPConnection();
	}

	private void handleFrame(RawFrame f) throws IOException, WebSocketException {
		// Validate it
		if (validate(f)) {
			// This frame must be handle
			try {
				handle(f);
			} catch (IllegalStateException e) {
				/*
				 * We can receive valid packets if the connection is not in the OPEN state. Example: someone ask to
				 * close the connection (user can do this because it chooses to do so, validate() can this because we
				 * receive an invalid frame) so the connection moves to the CLOSING state. At this moment, we can
				 * receive a ping frame. handle() will try to response with a pong frame and an IllegalstateException
				 * will be raised.
				 */
				Messages.LOGGER.log(Level.INFO, Messages.CATEGORY, Messages.VALID_PACKET_CONNECTION_CLOSED, e, this);
			}
		} else {
			// validate() took care to close the connection, simply wait for closing response and drop the current frame
			Messages.LOGGER.log(Level.WARNING, Messages.CATEGORY, Messages.INVALID_FRAME, this);
		}
	}

	/**
	 * Determine the opcode of a frame and take appropriate actions.
	 * <p>
	 * Section 10.7. Handling of Invalid Data provides interesting information.
	 *
	 * @param f
	 *            the frame to handle
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 * @throws IllegalStateException
	 *             this exception can be thrown by methods from {@link WebSocket} if the connection state is not OPEN
	 * @throws WebSocketException
	 *             If the frame is invalid.
	 */
	private void handle(RawFrame f) throws IOException, IllegalStateException, WebSocketException {
		switch (f.getOpcode()) {
		// Each opcode may lead to a different action

		case RawFrame.OPCODE_TEXT_FRAME:
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.RECEIVED_TEXT_FRAME, this);
			String text = new String(f.getPayload(true));
			String response = ws.getEndpoint().onTextMessage(this.ws, text);
			if (response != null) {
				ws.sendText(response);
			}
			break;

		case RawFrame.OPCODE_BINARY_FRAME:
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.RECEIVED_BINARY_FRAME, this);
			byte[] binary = f.getPayload(true);
			byte[] binaryResponse = ws.getEndpoint().onBinaryMessage(this.ws, binary);
			if (binaryResponse != null) {
				ws.sendBinary(binaryResponse);
			}
			break;

		case RawFrame.OPCODE_CONNECTION_CLOSE:
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.RECEIVED_CLOSE_FRAME, this);

			// Extract close code (section 5.5.1. Close)
			ReasonForClosure reasonForClosure = new ReasonForClosure(CloseCodes.NO_STATUS_PROVIDED, null);

			if (f.getLength() != 0) {
				byte[] payload = f.getPayload(true);

				// Close code MUST be the first two bytes
				int msb = payload[0] & 0xFF;
				int lsb = payload[1] & 0xFF;
				int code = (msb << 8) + lsb;

				// Close reason may by null (0 byte long)
				String reason = new String(f.getPayload(true), 2, payload.length - 2);

				reasonForClosure.setCode(code);
				reasonForClosure.setReason(reason);
			} // else
			// Section 7.1.5: if there is no close code then it is considered to be 1005
			// We don't set this code here because we cannot used this code in the close frame that we are going to send back
			// We will set this code just before calling Endpoint.onClose() callback

			// Properly react to the closing frame
			synchronized (ws) {
				if (ws.getCurrentState() == ConnectionStates.CLOSING) {
					// We started the closing handshake, we were expecting this response
					OnTimeOutCloser closer = ws.getOnTimeOutCloser();
					synchronized (closer) {
						ws.closeUnderlyingTCPConnection();
						closer.setClosed(true);
					}
				} else {
					// The remote endpoint has started the closing handshake
					ws.respondToClosingHandshake(reasonForClosure);
				}
			}

			ws.getEndpoint().onClose(ws, reasonForClosure);
			/*
			 * Section 7.1.5. The WebSocket Connection Close Code has an interesting note about both endpoints that start the closing handshake at
			 * roughly the same time. In such a case, they receive is a "start-the-closing-handhskake frame" but they think it is an anwer to their
			 * own frames. These frames may contain different close codes and consequently the endpoints will not agree on the websocket connection
			 * close code.
			 */
			break;

		case RawFrame.OPCODE_PING:
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.RECEIVED_PING_FRAME, this);
			/*
			 * Websocket.pong() documentation says it is for unsolicited pongs only. This comment is dedicated to end users of the Websocket library.
			 * Sending a pong when responding to a ping is the same as sending an unsolicited pong. Hence, we can use the same function here.
			 */
			ws.pong(f.getPayload(true));
			ws.getEndpoint().onPing(f.getPayload(true));
			break;

		case RawFrame.OPCODE_PONG:
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.RECEIVED_PONG_FRAME, this);
			ws.getEndpoint().onPong(f.getPayload(true));
			break;

		default:
			// WARNING: It is a bug to get in this default case!
			// The validate() method is here to catch invalid (as per RFC6455) or not managed (by this implementation)
			// opcodes.
			Messages.LOGGER.log(Level.SEVERE, Messages.CATEGORY, Messages.OPCODE_NOT_MANAGED, this);
			// Error in code: should not get in the default case
			throw new WebSocketException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.OPCODE_NOT_MANAGED));
		}
	}

	/**
	 * Validate if the frame is valid for this receiver. If not, this method will send a start the closing handshake (calling
	 * {@link WebSocket#close()}.
	 *
	 * For now, Receiver is intended for the client side of the connection (that is, frames not must be masked) and cannot manage fragmented frames.
	 * Furthermore, extensions are not managed so RSV bits must be 0.
	 *
	 * @param f
	 *            the frame to validate
	 * @return 'true' is the frame is valid and must be handled, 'false' is the frame is not valid and must be dropped
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 */
	private boolean validate(RawFrame f) throws IOException {
		try {
			/*
			 * Constraints from the RFC6455
			 */
			if (!f.isValid()) {
				ws.close(new ReasonForClosure(CloseCodes.PROTOCOL_ERROR, null));
				return false;
			}
		} catch (UnsupportedEncodingException e) {
			Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.INVALID_UTF8, e, this);
			ws.close(new ReasonForClosure(CloseCodes.RECEIVED_INCONSISTENT_DATA, e.getMessage()));
			return false;
		} catch (IOException e) {
			Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.ERROR_UNKNOWN, e, this);
			ws.close(new ReasonForClosure(CloseCodes.PROTOCOL_ERROR, e.getMessage()));
			return false;
		}

		/*
		 * Limitations of this implementation.
		 */
		// RSV must be 0
		if (f.getRSV() != 0) {
			// We must _Fail the WebSocket Connection_ (see page 28)
			ws.close(new ReasonForClosure(CloseCodes.PROTOCOL_ERROR, null));
			return false;
			// FIXME check against websocket parameter?
		}

		// Frame must not be masked
		if (f.isMasked()) {
			throw new UnsupportedOperationException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.UNSUPPORTED_MASKED));
		}

		// So far, so good
		return true;
	}
}
