/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.microej.microvg.test.TestUtilities.VEEPort;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.Matrix;
import ej.microvg.ResourceVectorImage;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorImage;

/**
 * Tests applying a color filter to an image.
 */
public class TestColorFilter {

	private static final String MASCOT = "/com/microej/microvg/test/mascot.xml";
	private static final String RED_RECTANGLE_SVG = "/com/microej/microvg/test/red_rectangle.xml";
	private static final String GRADIENT_RECTANGLE_AVD = "/com/microej/microvg/test/gradient.xml";
	private static final String ANIMATED_COLORS_AVD = "/com/microej/microvg/test/animated_colors.xml";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests passing a too short array.
	 */
	@Test
	public static void testColorFilterShortArray() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, // missing one alpha
		};
		boolean exception;
		try {
			VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
			exception = false;
		} catch (ArrayIndexOutOfBoundsException e) {
			exception = true;
		}

		Assert.assertTrue("exception", exception); //$NON-NLS-1$
	}

	/**
	 * Tests passing a too long array.
	 */
	@Test
	public static void testColorFilterLongArray() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
				0, // extra
		};
		boolean exception;
		try {
			VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
			exception = false;
		} catch (ArrayIndexOutOfBoundsException e) {
			exception = true;
		}

		Assert.assertFalse("exception", exception); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an overlapping image with a color matrix filter.
	 */
	@Test
	public static void testColorFilterOverlap() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(MASCOT);

		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		boolean exception;
		try {
			VectorGraphicsPainter.drawFilteredImage(g, image, new Matrix(), colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertFalse(exception);

		colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0, 0xff, // alpha
		};
		try {
			VectorGraphicsPainter.drawFilteredImage(g, image, new Matrix(), colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertFalse(exception);
	}

	/**
	 * Tests drawing an overlapping image with a color matrix filter that makes the color transparent.
	 */
	@Test
	public static void testColorFilterTransparentOverlap() {

		if (TestUtilities.isOn(VEEPort.Android) || TestUtilities.isOn(VEEPort.JavaFX)) {
			// On Android and JavaFX, it is allowed to draw an image with alpha + overlapping paths
			return;
		}

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(MASCOT);

		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0.5f, 0, // alpha
		};
		boolean exception;
		try {
			VectorGraphicsPainter.drawFilteredImage(g, image, new Matrix(), colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);

		colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0, 0x88, // alpha
		};
		try {
			VectorGraphicsPainter.drawFilteredImage(g, image, new Matrix(), colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);

		colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				-0.2f, 0.2f, 0, 1, 0, // alpha
		};
		try {
			VectorGraphicsPainter.drawFilteredImage(g, image, new Matrix(), colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);
	}

	/**
	 * Tests filtering an overlapping image with a color matrix filter.
	 */
	@Test
	public static void testColorFilterImageOverlap() {
		VectorImage image = VectorImage.getImage(MASCOT);

		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		boolean exception;
		try {
			image.filterImage(colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertFalse(exception);

		colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0, 0xff, // alpha
		};
		try {
			image.filterImage(colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertFalse(exception);
	}

	/**
	 * Tests filtering an overlapping image with a color matrix filter that makes the color transparent.
	 */
	@Test
	public static void testColorFilterImageTransparentOverlap() {

		if (TestUtilities.isOn(VEEPort.Android) || TestUtilities.isOn(VEEPort.JavaFX)) {
			// On Android and JavaFX, it is allowed to draw an image with alpha + overlapping paths
			return;
		}

		VectorImage image = VectorImage.getImage(MASCOT);

		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0.5f, 0, // alpha
		};
		boolean exception;
		try {
			image.filterImage(colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);

		colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0, 0x88, // alpha
		};
		try {
			image.filterImage(colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);

		colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				-0.2f, 0.2f, 0, 1, 0, // alpha
		};
		try {
			image.filterImage(colorMatrix);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);
	}

	/**
	 * Tests drawing an image with a color matrix filter changing nothing.
	 */
	@Test
	public static void testColorFilterIdentity() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, Colors.RED); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, Colors.RED); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, Colors.RED); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image with a color matrix filter changing the color by using the current one.
	 */
	@Test
	public static void testColorFilterUseColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0, // green
				0.5f, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, 0x7f007f); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0x7f007f); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0x7f007f); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image with a color matrix filter changing the color by setting a new one.
	 */
	@Test
	public static void testColorFilterNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0x88, // green
				0, 0, 0, 0, 0x88, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0x008888); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image with a color matrix filter modifying the alpha value, keeping the current color.
	 */
	@Test
	public static void testColorFilterAlpha() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0.4f, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, 0x660000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0x660000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0x660000); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image with a color matrix filter changing the colors with their grayscale equivalent.
	 */
	@Test
	public static void testColorFilterGrayscale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.299f, 0.587f, 0.114f, 0, 0, // red
				0.299f, 0.587f, 0.114f, 0, 0, // green
				0.299f, 0.587f, 0.114f, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, 0x4c4c4c); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0x4c4c4c); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0x4c4c4c); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image with a color matrix filter changing the color with components greater than 0xff.
	 */
	@Test
	public static void testColorFilterExceedMax() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0xff, // red
				0, 0, 0, 0, 0xffff, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, Colors.YELLOW); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, Colors.YELLOW); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, Colors.YELLOW); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image with a color matrix filter changing the color with components lower than 0x0.
	 */
	@Test
	public static void testColorFilterExceedMin() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				-1, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0xffffffff, // green
				0, 0, 0, 0, -0xff, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image with a color matrix filter changing the color with positive and negative values.
	 */
	@Test
	public static void testColorFilterPlusMinus() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				-0.5f, 0, 0, 0, 0xff, // red
				1, 0, 0, 0, -0x7f, // green
				0, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, 0x808000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0x808000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0x808000); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image after transforming it with a color matrix filter changing the color by using the current
	 * one.
	 */
	@Test
	public static void testColorFilterImageUseColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0, // green
				0.5f, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			VectorGraphicsPainter.drawImage(g, filterImage, matrix);
			display.flush();

			TestUtilities.check("top left", 50, 50, g, 0x7f007f); //$NON-NLS-1$
			TestUtilities.check("center", 99, 99, g, 0x7f007f); //$NON-NLS-1$
			TestUtilities.check("bottom right", 149, 149, g, 0x7f007f); //$NON-NLS-1$
		}
	}

	/**
	 * Tests drawing an image after transforming it with a color matrix filter changing the color by setting a new one.
	 */
	@Test
	public static void testColorFilterImageNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0x88, // green
				0, 0, 0, 0, 0x88, // blue
				0, 0, 0, 1, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			VectorGraphicsPainter.drawImage(g, filterImage, matrix);
			display.flush();

			TestUtilities.check("top left", 50, 50, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("center", 99, 99, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("bottom right", 149, 149, g, 0x008888); //$NON-NLS-1$
		}
	}

	/**
	 * Tests drawing an image after transforming it with a color matrix filter modifying the alpha value, keeping the
	 * current color.
	 */
	@Test
	public static void testColorFilterImageAlpha() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0.4f, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			VectorGraphicsPainter.drawImage(g, filterImage, matrix);
			display.flush();

			TestUtilities.check("top left", 50, 50, g, 0x660000); //$NON-NLS-1$
			TestUtilities.check("center", 99, 99, g, 0x660000); //$NON-NLS-1$
			TestUtilities.check("bottom right", 149, 149, g, 0x660000); //$NON-NLS-1$
		}
	}

	/**
	 * Tests drawing an image with a color matrix filter after transforming it with another color matrix filter.
	 */
	@Test
	public static void testColorFilterTwice() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix1 = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0x88, // green
				1, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		float[] colorMatrix2 = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0, 0.5f, 0, 0, 0, // green
				0, 0, 0, 0, 0x88, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorImage filterImage = image.filterImage(colorMatrix1);
		VectorGraphicsPainter.drawFilteredImage(g, filterImage, matrix, colorMatrix2);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, 0x3f4488); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0x3f4488); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0x3f4488); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an image with a color matrix filter after transforming it with another color matrix filter (the
	 * latter having only components multipliers).
	 */
	@Test
	public static void testColorFilterTwiceSimple() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(RED_RECTANGLE_SVG);

		Matrix matrix = new Matrix();
		float[] colorMatrix1 = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0.5f, 0, 0, 0, 0, // green
				1, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		float[] colorMatrix2 = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0, 0.5f, 0, 0, 0, // green
				0, 0, 0, 0, 0x88, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorImage filterImage = image.filterImage(colorMatrix1);
		VectorGraphicsPainter.drawFilteredImage(g, filterImage, matrix, colorMatrix2);
		display.flush();

		TestUtilities.check("top left", 50, 50, g, 0x3f3f88); //$NON-NLS-1$
		TestUtilities.check("center", 99, 99, g, 0x3f3f88); //$NON-NLS-1$
		TestUtilities.check("bottom right", 149, 149, g, 0x3f3f88); //$NON-NLS-1$
	}

	/**
	 * Tests drawing a gradient image with a color matrix filter changing nothing.
	 */
	@Test
	public static void testGradientColorFilterIdentity() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 0, 0, g, Colors.RED); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, Colors.LIME); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, Colors.BLUE); //$NON-NLS-1$
	}

	/**
	 * Tests drawing a gradient image with a color matrix filter changing the color by using the current one.
	 */
	@Test
	public static void testGradientColorFilterUseColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.5f, 0, 0.5f, 0, 0, // red
				0, 0, 0, 0, 0, // green
				0.5f, 0, 0.5f, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 0, 0, g, 0x7f007f); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0x7f007f); //$NON-NLS-1$
	}

	/**
	 * Tests drawing a gradient image with a color matrix filter changing the color by setting a new one.
	 */
	@Test
	public static void testGradientColorFilterNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0x88, // green
				0, 0, 0, 0, 0x88, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 0, 0, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0x008888); //$NON-NLS-1$
	}

	/**
	 * Tests drawing a gradient image with a color matrix filter modifying the alpha value, keeping the current color.
	 */
	@Test
	public static void testGradientColorFilterAlpha() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0.4f, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 0, 0, g, 0x660000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0x006600); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0x000066); //$NON-NLS-1$
	}

	/**
	 * Tests drawing a gradient image with a color matrix filter changing the colors with their grayscale equivalent.
	 */
	@Test
	public static void testGradientColorFilterGrayscale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.299f, 0.587f, 0.114f, 0, 0, // red
				0.299f, 0.587f, 0.114f, 0, 0, // green
				0.299f, 0.587f, 0.114f, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
		display.flush();

		TestUtilities.check("top left", 0, 0, g, 0x4c4c4c); //$NON-NLS-1$
		TestUtilities.check("center", 49, 49, g, 0x969696); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0x1d1d1d); //$NON-NLS-1$
	}

	/**
	 * Tests drawing a gradient image after transforming it with a color matrix filter changing the color by using the
	 * current one.
	 */
	@Test
	public static void testGradientColorFilterImageUseColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.5f, 0, 0.5f, 0, 0, // red
				0, 0, 0, 0, 0, // green
				0.5f, 0, 0.5f, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			VectorGraphicsPainter.drawImage(g, filterImage, matrix);
			display.flush();

			TestUtilities.check("top left", 0, 0, g, 0x7f007f); //$NON-NLS-1$
			TestUtilities.check("center", 49, 49, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			TestUtilities.check("bottom right", 99, 99, g, 0x7f007f); //$NON-NLS-1$
		}
	}

	/**
	 * Tests drawing a gradient image after transforming it with a color matrix filter changing the color by setting a
	 * new one.
	 */
	@Test
	public static void testGradientColorFilterImageNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0x88, // green
				0, 0, 0, 0, 0x88, // blue
				0, 0, 0, 1, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			VectorGraphicsPainter.drawImage(g, filterImage, matrix);
			display.flush();

			TestUtilities.check("top left", 0, 0, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("center", 49, 49, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("bottom right", 99, 99, g, 0x008888); //$NON-NLS-1$
		}
	}

	/**
	 * Tests drawing a gradient image after transforming it with a color matrix filter modifying the alpha value,
	 * keeping the current color.
	 */
	@Test
	public static void testGradientColorFilterImageAlpha() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0.4f, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			VectorGraphicsPainter.drawImage(g, filterImage, matrix);
			display.flush();

			TestUtilities.check("top left", 0, 0, g, 0x660000); //$NON-NLS-1$
			TestUtilities.check("center", 49, 49, g, 0x006600); //$NON-NLS-1$
			TestUtilities.check("bottom right", 99, 99, g, 0x000066); //$NON-NLS-1$
		}
	}

	/**
	 * Tests passing a too short array.
	 */
	@Test
	public static void testAnimatedColorFilterShortArray() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, // missing one alpha
		};
		boolean exception;
		try {
			VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
			exception = false;
		} catch (ArrayIndexOutOfBoundsException e) {
			exception = true;
		}

		Assert.assertTrue("exception", exception); //$NON-NLS-1$
	}

	/**
	 * Tests passing a too long array.
	 */
	@Test
	public static void testAnimatedColorFilterLongArray() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
				0, // extra
		};
		boolean exception;
		try {
			VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
			exception = false;
		} catch (ArrayIndexOutOfBoundsException e) {
			exception = true;
		}

		Assert.assertFalse("exception", exception); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image with a color matrix filter changing nothing.
	 */
	@Test
	public static void testAnimatedColorFilterIdentity() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		// Test at start of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, Colors.RED); //$NON-NLS-1$
		TestUtilities.check("start - strokeColor", 100, 0, g, Colors.RED); //$NON-NLS-1$
		TestUtilities.check("start - fillAlpha", 0, 100, g, Colors.LIME); //$NON-NLS-1$
		TestUtilities.check("start - strokeAlpha", 100, 100, g, Colors.LIME); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, image.getDuration(), colorMatrix);
		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, Colors.BLUE); //$NON-NLS-1$
		TestUtilities.check("end - strokeColor", 100, 0, g, Colors.BLUE); //$NON-NLS-1$
		TestUtilities.check("end - fillAlpha", 0, 100, g, 0x008000); //$NON-NLS-1$
		TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x008000); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image with a color matrix filter changing the color by using the current one.
	 */
	@Test
	public static void testAnimatedColorFilterUseColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0, 0.5f, 0, 0, 0, // green
				0.5f, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		// Test at start of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, 0x7f007f); //$NON-NLS-1$
		TestUtilities.check("start - strokeColor", 100, 0, g, 0x7f007f); //$NON-NLS-1$
		TestUtilities.check("start - fillAlpha", 0, 100, g, 0x007f00); //$NON-NLS-1$
		TestUtilities.check("start - strokeAlpha", 100, 100, g, 0x007f00); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, image.getDuration(), colorMatrix);
		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("end - strokeColor", 100, 0, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("end - fillAlpha", 0, 100, g, 0x004000); //$NON-NLS-1$
		TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x004000); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image with a color matrix filter changing the color by setting a new one.
	 */
	@Test
	public static void testAnimatedColorFilterNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0x88, // green
				0, 0, 0, 0, 0x88, // blue
				0, 0, 0, 1, 0, // alpha
		};
		// Test at start of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("start - strokeColor", 100, 0, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("start - fillAlpha", 0, 100, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("start - strokeAlpha", 100, 100, g, 0x008888); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, image.getDuration(), colorMatrix);
		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("end - strokeColor", 100, 0, g, 0x008888); //$NON-NLS-1$
		TestUtilities.check("end - fillAlpha", 0, 100, g, 0x004444); //$NON-NLS-1$
		TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x004444); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image with a color matrix filter modifying the alpha value, keeping the current color.
	 */
	@Test
	public static void testAnimatedColorFilterAlpha() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0.4f, 0, // alpha
		};
		// Test at start of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, 0x660000); //$NON-NLS-1$
		TestUtilities.check("start - strokeColor", 100, 0, g, 0x660000); //$NON-NLS-1$
		TestUtilities.check("start - fillAlpha", 0, 100, g, 0x006600); //$NON-NLS-1$
		TestUtilities.check("start - strokeAlpha", 100, 100, g, 0x006600); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, image.getDuration(), colorMatrix);
		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, 0x000066); //$NON-NLS-1$
		TestUtilities.check("end - strokeColor", 100, 0, g, 0x000066); //$NON-NLS-1$
		TestUtilities.check("end - fillAlpha", 0, 100, g, 0x003300); //$NON-NLS-1$
		TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x003300); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image with a color matrix filter changing the colors with their grayscale equivalent.
	 */
	@Test
	public static void testAnimatedColorFilterGrayscale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.299f, 0.587f, 0.114f, 0, 0, // red
				0.299f, 0.587f, 0.114f, 0, 0, // green
				0.299f, 0.587f, 0.114f, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		// Test at start of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, 0x4c4c4c); //$NON-NLS-1$
		TestUtilities.check("start - strokeColor", 100, 0, g, 0x4c4c4c); //$NON-NLS-1$
		TestUtilities.check("start - fillAlpha", 0, 100, g, 0x969696); //$NON-NLS-1$
		TestUtilities.check("start - strokeAlpha", 100, 100, g, 0x969696); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, image.getDuration(), colorMatrix);
		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, 0x1d1d1d); //$NON-NLS-1$
		TestUtilities.check("end - strokeColor", 100, 0, g, 0x1d1d1d); //$NON-NLS-1$
		TestUtilities.check("end - fillAlpha", 0, 100, g, 0x4b4b4b); //$NON-NLS-1$
		TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x4b4b4b); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image with a color matrix filter changing the color with components greater than 0xff.
	 */
	@Test
	public static void testAnimatedColorFilterExceedMax() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0xff, // red
				0, 0, 0, 0, 0xffff, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		// Test at start of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, Colors.YELLOW); //$NON-NLS-1$
		TestUtilities.check("start - strokeColor", 100, 0, g, Colors.YELLOW); //$NON-NLS-1$
		TestUtilities.check("start - fillAlpha", 0, 100, g, Colors.YELLOW); //$NON-NLS-1$
		TestUtilities.check("start - strokeAlpha", 100, 100, g, Colors.YELLOW); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, image.getDuration(), colorMatrix);
		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, Colors.WHITE); //$NON-NLS-1$
		TestUtilities.check("end - strokeColor", 100, 0, g, Colors.WHITE); //$NON-NLS-1$
		TestUtilities.check("end - fillAlpha", 0, 100, g, 0x808000); //$NON-NLS-1$
		TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x808000); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image with a color matrix filter changing the color with components lower than 0x0.
	 */
	@Test
	public static void testAnimatedColorFilterExceedMin() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				-1, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0xffffffff, // green
				0, 0, 0, 0, -0xff, // blue
				0, 0, 0, 1, 0, // alpha
		};
		// Test at start of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("start - strokeColor", 100, 0, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("start - fillAlpha", 0, 100, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("start - strokeAlpha", 100, 100, g, TestUtilities.BACKGROUND_COLOR); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, image.getDuration(), colorMatrix);
		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("end - strokeColor", 100, 0, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("end - fillAlpha", 0, 100, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
		TestUtilities.check("end - strokeAlpha", 100, 100, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image with a color matrix filter changing the color with positive and negative values.
	 */
	@Test
	public static void testAnimatedColorFilterPlusMinus() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				-0.5f, 0, 0, 0, 0xff, // red
				0, 1, 0, 0, -0x7f, // green
				0, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		// Test at start of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, 0, colorMatrix);
		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, 0x7f0000); //$NON-NLS-1$
		TestUtilities.check("start - strokeColor", 100, 0, g, 0x7f0000); //$NON-NLS-1$
		TestUtilities.check("start - fillAlpha", 0, 100, g, 0xff8000); //$NON-NLS-1$
		TestUtilities.check("start - strokeAlpha", 100, 100, g, 0xff8000); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawFilteredAnimatedImage(g, image, matrix, image.getDuration(), colorMatrix);
		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, Colors.RED); //$NON-NLS-1$
		TestUtilities.check("end - strokeColor", 100, 0, g, Colors.RED); //$NON-NLS-1$
		TestUtilities.check("end - fillAlpha", 0, 100, g, 0x7f4000); //$NON-NLS-1$
		TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x7f4000); //$NON-NLS-1$
	}

	/**
	 * Tests drawing an animated image after transforming it with a color matrix filter changing the color by using the
	 * current one.
	 */
	@Test
	public static void testAnimatedColorFilterImageUseColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0, 0.5f, 0, 0, 0, // green
				0.5f, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			// Test at start of animation
			VectorGraphicsPainter.drawAnimatedImage(g, filterImage, matrix, 0);
			display.flush();

			TestUtilities.check("start - fillColor", 0, 0, g, 0x7f007f); //$NON-NLS-1$
			TestUtilities.check("start - strokeColor", 100, 0, g, 0x7f007f); //$NON-NLS-1$
			TestUtilities.check("start - fillAlpha", 0, 100, g, 0x007f00); //$NON-NLS-1$
			TestUtilities.check("start - strokeAlpha", 100, 100, g, 0x007f00); // $NON-NLS-1

			g.setColor(TestUtilities.BACKGROUND_COLOR);
			Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

			// Test at end of animation
			VectorGraphicsPainter.drawAnimatedImage(g, filterImage, matrix, image.getDuration());
			display.flush();

			TestUtilities.check("end - fillColor", 0, 0, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			TestUtilities.check("end - strokeColor", 100, 0, g, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			TestUtilities.check("end - fillAlpha", 0, 100, g, 0x004000); //$NON-NLS-1$
			TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x004000); //$NON-NLS-1$
		}
	}

	/**
	 * Tests drawing an animated image after transforming it with a color matrix filter changing the color by setting a
	 * new one.
	 */
	@Test
	public static void testAnimatedColorFilterImageNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				0, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0x88, // green
				0, 0, 0, 0, 0x88, // blue
				0, 0, 0, 1, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			// Test at start of animation
			VectorGraphicsPainter.drawAnimatedImage(g, filterImage, matrix, 0);
			display.flush();

			TestUtilities.check("start - fillColor", 0, 0, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("start - strokeColor", 100, 0, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("start - fillAlpha", 0, 100, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("start - strokeAlpha", 100, 100, g, 0x008888); // $NON-NLS-1

			g.setColor(TestUtilities.BACKGROUND_COLOR);
			Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

			// Test at end of animation
			VectorGraphicsPainter.drawAnimatedImage(g, filterImage, matrix, image.getDuration());
			display.flush();

			TestUtilities.check("end - fillColor", 0, 0, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("end - strokeColor", 100, 0, g, 0x008888); //$NON-NLS-1$
			TestUtilities.check("end - fillAlpha", 0, 100, g, 0x004444); //$NON-NLS-1$
			TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x004444); //$NON-NLS-1$
		}
	}

	/**
	 * Tests drawing an animated image after transforming it with a color matrix filter modifying the alpha value,
	 * keeping the current color.
	 */
	@Test
	public static void testAnimatedColorFilterImageAlpha() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		Matrix matrix = new Matrix();
		float[] colorMatrix = new float[] { //
				1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 0.4f, 0, // alpha
		};
		try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
			// Test at start of animation
			VectorGraphicsPainter.drawAnimatedImage(g, filterImage, matrix, 0);
			display.flush();

			TestUtilities.check("start - fillColor", 0, 0, g, 0x660000); //$NON-NLS-1$
			TestUtilities.check("start - strokeColor", 100, 0, g, 0x660000); //$NON-NLS-1$
			TestUtilities.check("start - fillAlpha", 0, 100, g, 0x006600); //$NON-NLS-1$
			TestUtilities.check("start - strokeAlpha", 100, 100, g, 0x006600); // $NON-NLS-1

			g.setColor(TestUtilities.BACKGROUND_COLOR);
			Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

			// Test at end of animation
			VectorGraphicsPainter.drawAnimatedImage(g, filterImage, matrix, image.getDuration());
			display.flush();

			TestUtilities.check("end - fillColor", 0, 0, g, 0x000066); //$NON-NLS-1$
			TestUtilities.check("end - strokeColor", 100, 0, g, 0x000066); //$NON-NLS-1$
			TestUtilities.check("end - fillAlpha", 0, 100, g, 0x003300); //$NON-NLS-1$
			TestUtilities.check("end - strokeAlpha", 100, 100, g, 0x003300); //$NON-NLS-1$
		}
	}

}
