/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.microai;

/**
 * The <code>OutputTensor</code> class offers services to deal with MicroAI output tensors.
 * <p>
 * <code>OutputTensor</code> is an output of your {@link MLInferenceEngine}.
 */
public class OutputTensor extends Tensor {
    private final int modelHandle;
    private final int tensorIndex;
    OutputTensor(int modelHandle, int index) {
        super(modelHandle, index, false);
        this.modelHandle = modelHandle;
        this.tensorIndex = index;
    }

    /**
     * Gets the output data of the tensor.
     * <p>
     * <code>outputData</code> is an array of signed or unsigned bytes.
     *
     * @param outputData
     *            the inference result as byte array.
     */
    public void getOutputData(byte[] outputData) {
        MLNatives.getOutputDataAsByteArray(this.modelHandle, this.tensorIndex, outputData);
    }

    /**
     * Gets the output data of the tensor.
     * <p>
     * <code>outputData</code> is an array of signed or unsigned integers.
     *
     * @param outputData
     *            the inference result as integer array.
     */
    public void getOutputData(int[] outputData) {
        MLNatives.getOutputDataAsIntegerArray(this.modelHandle, this.tensorIndex, outputData);
    }

    /**
     * Gets the output data of the tensor.
     * <p>
     * <code>outputData</code> is a float array.
     *
     * @param outputData
     *            the inference result as float array.
     */
    public void getOutputData(float[] outputData) {
        MLNatives.getOutputDataAsFloatArray(this.modelHandle, this.tensorIndex, outputData);
    }
}
