/*
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.observable;

import ej.annotation.Nullable;

/**
 * This class represents an observable variant which can handle only a single observer.
 */
public class SimpleObservable {

	private @Nullable Observer observer;
	private boolean changed;

	/**
	 * Creates a simple observable.
	 */
	public SimpleObservable() {
		this.observer = null;
	}

	/**
	 * Returns the observer of this simple observable, if it is set.
	 *
	 * @return the observer of this simple observable, or <code>null</code> if it is not set.
	 */
	public @Nullable Observer getObserver() {
		return this.observer;
	}

	/**
	 * Sets the observer of this simple observable.
	 *
	 * @param observer
	 *            the observer to set.
	 * @throws NullPointerException
	 *             if the given observer is <code>null</code>.
	 */
	@SuppressWarnings({ "null", "unused" })
	public void setObserver(Observer observer) {
		if (observer == null) {
			throw new NullPointerException();
		}
		this.observer = observer;
	}

	/**
	 * Unsets the observer of this simple observable if it the same instance as the given observer.
	 *
	 * @param observer
	 *            the observer to unset.
	 */
	public void unsetObserver(Observer observer) {
		if (observer == this.observer) {
			this.observer = null;
		}
	}

	/**
	 * Unsets the observer of this simple observable.
	 */
	public void unsetObserver() {
		this.observer = null;
	}

	/**
	 * Marks this simple observable as having been changed. The {@link #hasChanged()} method will now return
	 * <code>true</code>.
	 */
	protected void setChanged() {
		this.changed = true;
	}

	/**
	 * Indicates that this simple observable has no longer changed, or that it has already notified all of its observers
	 * of its most recent change, so that the {@link #hasChanged()} method will now return <code>false</code>. This
	 * method is called automatically by the {@link #notifyObserver()} method.
	 */
	protected void clearChanged() {
		this.changed = false;
	}

	/**
	 * Gets whether or not this simple observable has changed.
	 *
	 * @return <code>true</code> if and only if the {@link #setChanged()} method has been called more recently than the
	 *         {@link #clearChanged()} method on this simple observable, <code>false</code> otherwise.
	 */
	public boolean hasChanged() {
		return this.changed;
	}

	/**
	 * Notifies the registered observer, if any.
	 */
	public void notifyObserver() {
		if (!this.changed) {
			return;
		}
		clearChanged();

		Observer observer = this.observer;
		if (observer != null) {
			observer.update();
		}
	}
}
