/*
 * Copyright 2014-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.websocket;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.util.Base64;
import java.util.Random;

import ej.basictool.ArrayTools;
import ej.util.message.Level;
import ej.websocket.frame.ClientFrameBuilder;
import ej.websocket.frame.RawFrame;
import ej.websocket.http.Header;
import ej.websocket.http.HttpResponse;

/**
 * Representation for the physical websocket connection following RFC 6455 definition. It needs a {@link WebSocketURI}
 * that describes the server to connect and an {@link Endpoint} to handle events (errors, incoming data). A
 * {@link WebSocket} is the physical medium, managing underlying TCP connection and low level websocket events, whereas
 * the {@link Endpoint} is here to handle high level events. The latter is more application oriented whereas the former
 * is the physical medium.
 * <p>
 * WARNING: this implementation is designed so that {@link WebSocket} is only for client-side connection.
 *
 *
 *
 */
public class WebSocket implements AutoCloseable {

	// EOL for the opening handshake
	private static final String HTTP_END_OF_LINE = "\r\n";

	// RFC 6455 defines the version 13 for websocket
	private static final String RFC_WEBSOCKET_VERSION = "13";

	// 16 bytes random number that has been Base64 encoded (see 4.1 point 7 of the second list)
	private static final int RFC_NONCE_KEY_LENGTH = 16;

	// Underlying TCP connection (socket and associated I/O streams)
	private InputStream is;
	private Socket socket;
	private OutputStream os;

	// Current state of this websocket
	private ConnectionStates currentState;

	// URI of the remote endpoint
	private final WebSocketURI uri;

	// Custom opening handshake headers
	private String[] customHeaders;

	// Applicative handler for events occurring on this websocket
	private final Endpoint endpoint;

	// It will build appropriate frames
	private final ClientFrameBuilder frameBuilder;

	// This runnable delay the TCP disconnection when close() is called
	private OnTimeOutCloser onTimeOutCloser;

	/**
	 * Create a new {@link WebSocket} instance. It doesn't connect to the remote server; user has to call
	 * {@link #connect()} to do so.
	 *
	 * @param uri
	 *            URI to connect to
	 * @param endpoint
	 *            the endpoint that will handle events
	 * @throws NullPointerException
	 *             if 'endpoint' or 'uri' is null
	 */
	public WebSocket(WebSocketURI uri, Endpoint endpoint) throws NullPointerException {
		// We cannot work if the endpoint is null
		if (endpoint == null) {
			throw new NullPointerException();
		}

		// We cannot work if the URI is null
		// Here, we are sure that the URI is valid because the WebSocketURI would have failed otherwise.
		if (uri == null) {
			// URI is invalid (or null)
			throw new NullPointerException();
		}

		// Save our fields
		this.uri = uri;
		this.customHeaders = new String[0];
		this.endpoint = endpoint;
		this.currentState = ConnectionStates.NEW;
		this.frameBuilder = new ClientFrameBuilder();
	}

	/**
	 * Adds a header to send when connecting.
	 *
	 * @param name
	 *            the header name
	 * @param value
	 *            the header value
	 */
	public void addHeader(String name, String value) {
		String header = name + ": " + value;
		this.customHeaders = ArrayTools.add(this.customHeaders, header);
	}

	/**
	 * Establish a connection to the server described by the URI. Once the connection is up, the endpoint will be notify of high level events whereas
	 * the websocket will manage low level events (such as ping, close, etc) thank a thread running a {@link Receiver}. If at some point a problem is
	 * found, this method will throw a {@link WebSocketException}.
	 * <p>
	 * See section 4.1. Client Requirements.
	 *
	 * @throws IllegalArgumentException
	 *             if the current state of this websocket is not NEW
	 * @throws ServerException
	 *             if the connection is established but the server doesn't answer properly
	 * @throws WebSocketException
	 *             if an error occurs during connection (most probably a socket error)
	 */
	public void connect() throws IllegalArgumentException, WebSocketException, ServerException {
		// Be sure that this is the first call to connect() on this object
		if (currentState != ConnectionStates.NEW) {
			throw new IllegalStateException();
		}

		// The following steps are from the RFC6455

		// 1) Validate the URI is required by section 4.1. Client Requirements (point 1, page 15).
		// --> Done in constructor

		// FIXME 2) There must be only one CONNECTING websocket at the same time to the same host (point 2, page 15)

		// Section 4.1: "A connection is defined to initially be in a CONNECTING state"
		currentState = ConnectionStates.CONNECTING;

		// 3) "_Proxy Usage_"
		// TODO manage proxy
		// Open TCP connection (socket, I/O streams)
		try {
			setupSocket();
		} catch (IOException e) {
			// 4) _Fail the WebSocket Connection_. At this point, it is just
			// closing the TCP connection.
			closeUnderlyingTCPConnection();
			throw new WebSocketException(e);
		}


		// From this point, the TCP connection to the server is established "including a connection via a proxy or over a TLS-encrypted tunnel".

		// Perform opening handshake (page 17)
		performOpeningHandshake();

		// The "WebSocket Connection is Established" and can now move to the OPEN state (see end of section 4.1, page 20)
		currentState = ConnectionStates.OPEN;

		// Websocket can now operate a full duplex communication
		new Thread(new Receiver(this), "Websocket Receiver").start();
		this.endpoint.onOpen(this);
	}

	/**
	 * Open the socket and sets it.
	 *
	 * @throws IOException
	 *             if an {@link IOException} occurs.
	 * @see WebSocket#setSocket(Socket)
	 */
	protected void setupSocket() throws IOException {
		if (this.socket == null) {
			this.setSocket(new Socket(uri.getHost(), uri.getPort()));
		}
	}

	/**
	 * Sets the socket.
	 *
	 * @param socket the socket to set.
	 * @throws IOException
	 */
	protected void setSocket(Socket socket) throws IOException {
		this.socket = socket;
		is = new BufferedInputStream(socket.getInputStream());
		os = socket.getOutputStream();
	}

	/**
	 * Returns the current state of this {@link WebSocket}, see
	 * {@link ConnectionStates} for the returned states.
	 *
	 * @return the current state of this {@link WebSocket}
	 */
	public ConnectionStates getCurrentState() {
		return currentState;
	}

	/**
	 * Returns the {@link Endpoint} associated to this {@link WebSocket}.
	 *
	 * @return the {@link Endpoint} associated to this {@link WebSocket}
	 */
	public Endpoint getEndpoint() {
		return endpoint;
	}

	/**
	 * @return the {@link InputStream} of the underlying TCP connection.
	 */
	/* default */InputStream getInputStream() {
		return is;
	}

	/**
	 * @return the {@link OnTimeOutCloser} on this websocket; can be null if {@link #close()} has not been called yet.
	 */
	/* default */OnTimeOutCloser getOnTimeOutCloser() {
		return onTimeOutCloser;
	}

	/**
	 * @return the {@link WebSocketURI} of the remote endpoint
	 */
	public WebSocketURI getURI() {
		return uri;
	}

	/**
	 * Close the websocket connection as described in Section 7. Closing the Connection..
	 * <ul>
	 * <li>Send a closing handshake to the server (see Section 7.1.2. Start the WebSocket Closing Handshake).
	 * <li>Start a thread that will close the TCP connection after a timeout if the server doesn't answer thanks to a {@link OnTimeOutCloser}.
	 * <li>Wait for a close frame in the Receiver to close the TCP connection and stop the OnTimeOutCloser thread.
	 * </ul>
	 * <p>
	 * As described in section 7.1.1 of the RFC, the client should ask the server to close the underlying TCP connection. If the client closes the TCP
	 * connection first, it would prevent it from re-opening the connection.
	 *
	 * @param reasonForClosure
	 *            the reason for closing the connection that will be sent to remote endpoint
	 * @throws IllegalStateException
	 *             if the connection state is not OPEN
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 *
	 * @see OnTimeOutCloser
	 */
	public synchronized void close(ReasonForClosure reasonForClosure) throws IOException {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.CLOSE, this, reasonForClosure);

		checkOpen();

		// Section 7.1.2: start the websocket closing handshake
		RawFrame f = frameBuilder.buildCloseFrame(reasonForClosure);
		os.write(f.getBytes());

		// Section 7.1.3: move to CLOSING state
		currentState = ConnectionStates.CLOSING;

		// Timeout before closing the TCP connection if the server doesn't answer
		onTimeOutCloser = new OnTimeOutCloser(this);
		onTimeOutCloser.schedule();

		// Endpoint.onClose() will be called by the Receiver when a close frame is received or by the OnTimeOutCloser when timeout expires,
		// so we don't call it here.
	}

	/**
	 * Equivalent to <code>close(new ReasonForClosure(CloseCodes.NORMAL_CLOSURE, ""))</code>.
	 *
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 */
	@Override
	public synchronized void close() throws IOException {
		try {
			close(new ReasonForClosure(CloseCodes.NORMAL_CLOSURE, null));
		} catch (IOException e) {
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.ALREADY_CLOSE, e, this);
		}
	}

	/**
	 * Send ping to remote endpoint.
	 *
	 * @param payload
	 *            the payload of the ping
	 *
	 * @throws IllegalStateException
	 *             if the connection state is not OPEN
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 */
	public synchronized void ping(byte[] payload) throws IllegalStateException, IOException {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.PING, this);

		checkOpen();

		// See Section 5.5.2 Ping
		RawFrame f = frameBuilder.buildPingFrame(payload);
		os.write(f.getBytes());
	}

	/**
	 * Send ping with no payload to remote endpoint.
	 *
	 * @throws IllegalStateException
	 *             if the connection state is not OPEN
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 */
	public synchronized void ping() throws IllegalStateException, IOException {
		this.ping(null);
	}

	/**
	 * Send an unsolicited pong to remote endpoint.
	 *
	 * @param payload
	 *            the payload of the pong
	 *
	 * @throws IllegalStateException
	 *             if the connection state is not OPEN
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 */
	public synchronized void pong(byte[] payload) throws IllegalStateException, IOException {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.PONG, this);

		checkOpen();

		// See Section 5.5.3 Pong
		RawFrame f = frameBuilder.buildPongFrame(payload);
		os.write(f.getBytes());
	}

	/**
	 * Send an unsolicited pong with no payload to remote endpoint.
	 *
	 * @throws IllegalStateException
	 *             if the connection state is not OPEN
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 */
	public synchronized void pong() throws IllegalStateException, IOException {
		this.pong(null);
	}

	/**
	 * Sends a binary message.
	 *
	 * @param binary binary message to send
	 * @throws IllegalStateException if the connection state is not OPEN
	 * @throws IOException           thrown by the underlying {@link Socket} and
	 *                               associated streams when an error occurs
	 */
	public synchronized void sendBinary(byte[] binary) throws IllegalStateException, IOException {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.SEND_BINARY, this, binary);

		// See Section 6.1. Sending Data
		checkOpen();
		RawFrame f = frameBuilder.buildBinaryFrame(binary);
		try {
			os.write(f.getBytes());
		} catch (IOException e) {
			closeUnderlyingTCPConnection();
			this.endpoint.onClose(this, new ReasonForClosure(CloseCodes.CONNECTION_CLOSED_ABNORMALLY, ""));
			throw e;
		}
	}

	/**
	 * Sends a text message.
	 * <p>
	 * Warning: by default, MicroEJ strings are not encoded in UTF-8, to send an
	 * UTF-8 string you need to declare it as follow
	 * <code>String s = new String("yourString".getBytes("UTF-8);</code>
	 *
	 * @param text text message to send
	 * @throws IllegalStateException if the connection state is not OPEN
	 * @throws IOException           thrown by the underlying {@link Socket} and
	 *                               associated streams when an error occurs
	 */
	public synchronized void sendText(String text) throws IllegalStateException, IOException {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.SEND_TEXT, this, text);

		// See Section 6.1. Sending Data
		checkOpen();
		RawFrame f = frameBuilder.buildTextFrame(text);
		try {
			os.write(f.getBytes());
		} catch (IOException e) {
			closeUnderlyingTCPConnection();
			this.endpoint.onClose(this, new ReasonForClosure(CloseCodes.CONNECTION_CLOSED_ABNORMALLY, ""));
			throw e;
		}
	}

	/**
	 * Respond to a closing handshake and then close the underlying TCP connection (by calling {@link #closeUnderlyingTCPConnection()}).
	 *
	 * @param reasonForClosure
	 *            should be the reason received from the remote endpoint
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 */
	/* default */void respondToClosingHandshake(ReasonForClosure reasonForClosure) throws IOException {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.RESPOND_CLOSING, this);

		RawFrame f = frameBuilder.buildCloseFrame(reasonForClosure);
		os.write(f.getBytes());

		// Move to CLOSING state (see Section 7.1.3. The WebSocket Closing Handshake is Started)
		currentState = ConnectionStates.CLOSING;

		closeUnderlyingTCPConnection();
	}

	/**
	 * Close the underlying TCP connection and move the websocket to the CLOSED state.
	 * <p>
	 * This function must be called to terminate connection closure. It check if the thread managing the timeout is running and stops it if necessary.
	 */
	/* default */void closeUnderlyingTCPConnection() {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.CLOSE_TCP, this);

		// Websocket is now closed
		currentState = ConnectionStates.CLOSED;
		/*
		 * It may seem more logicial to move to the CLOSED state at the end of this method. That's true. But: when we close the resources from another
		 * thread that the Receiver's thread (namely, OnTimeOutCloser's thread), the Receiver may get an IOException because it is blocked on a read()
		 * and closing the input stream will unblock the read() with an IOException. When the Receiver.run() get such an IOException, it checks the
		 * state of the current state of the websocket to determine whether the exception is really an error or not . Indeed, if the websocket
		 * connection is closed to the TCP connection is supposed to be closed too.
		 */

		/*
		 * We try to close each resource once. If an IOException is raised, we assume that we cannot take any corrective action and that the resource
		 * has been released. Note that this function is called when the TCP connection fails to establish and that in this particular, the resources
		 * may be null.
		 */
		// Input stream
		try {
			if (is != null) {
				is.close();
			}
		} catch (IOException e) {
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.ERROR_UNKNOWN, this, e);
		}

		// Output stream
		try {
			if (os != null) {
				os.close();
			}
		} catch (IOException e) {
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.ERROR_UNKNOWN, this, e);
		}

		// Socket
		try {
			if (socket != null) {
				socket.close();
			}
		} catch (IOException e) {
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.ERROR_UNKNOWN, this, e);
		}
	}

	/**
	 * Check that the websocket current state is OPEN. If not
	 *
	 * @throws IOException
	 *             if the connection is not open.
	 */
	private void checkOpen() throws IOException {
		if (currentState != ConnectionStates.OPEN) {
			throw new IOException();
		}
	}

	private boolean isUsingDefaultPort() {
		return uri.getPort() == WebSocketURI.DEFAULT_NOT_SECURE_PORT;
		// TODO future versions should also check against the default port for wss
	}

	/**
	 * Perform the opening handshake.
	 *
	 * @throws ServerException
	 *             if the server doesn't answer properly
	 * @throws WebSocketException
	 *             if the handshake fails for any other reason (most probably a socket error)
	 */
	private void performOpeningHandshake() throws WebSocketException, ServerException {
		// Nonce is a random 16-byte value. See section 4.1, page 18, bullet 7
		// It is sent to server in the opening handshake
		// Server answer is validated using the same nonce
		byte[] nonceBytes = new byte[RFC_NONCE_KEY_LENGTH];
		new Random(System.currentTimeMillis()).nextBytes(nonceBytes);
		String nonce = Base64.getEncoder().encodeToString(nonceBytes);

		// Send handshake to server
		try {
			sendOpeningHandshake(nonce);
		} catch (IOException e) {
			closeUnderlyingTCPConnection();
			// Error while sending opening handshake
			throw new WebSocketException(e);
		}

		// Receive server's response
		HttpResponse response;
		try {
			response = new HttpResponse(is);
			Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.RECEIVE, this, response);
		} catch (IOException e) {
			closeUnderlyingTCPConnection();
			// Error while receiving server response to handshake
			throw new WebSocketException(e);
		}

		// Validate this response
		if (!validateResponse(response, nonce)) {
			// It it is not valid, we must _Fail the WebSocket Connection_ (see list page 19).
			// The connection has not been fully upgraded so we can simply close the TCP connection.
			closeUnderlyingTCPConnection();
			// Server did not accept the websocket connection or did not answer properly
			ServerException se = new ServerException();
			int serverStatusCode = response.getStatusLine().getStatusCode();
			se.setHttpStatusCode(serverStatusCode);
			throw se;
		}

	}

	/**
	 * Send the opening handshake to the server as described in the RFC6455.
	 *
	 * @param nonce
	 *            nonce as described if the RFC (random 16-byte value, as a base64 encoded string)
	 * @throws IOException
	 *             thrown by the underlying {@link Socket} and associated streams when an error occurs
	 */
	private void sendOpeningHandshake(String nonce) throws IOException {
		Messages.LOGGER.log(Level.FINE, Messages.CATEGORY, Messages.OPENING_HANDSHAKE, this);

		// TODO futures versions may use an HttpMessage class to construct the request

		// Form client handshake according to the context of the connection
		StringBuilder sb = new StringBuilder();

		// GET
		sb.append("GET ");
		sb.append(uri.getResourceName());
		sb.append(" HTTP/1.1");
		sb.append(HTTP_END_OF_LINE);

		// Host
		sb.append("Host: ");
		sb.append(uri.getHost());
		if (!isUsingDefaultPort()) {
			// RFC6455 page 17, point 4:
			// "|Host| header field whose value contains /host/ plus optionally ":" followed by /port/ (when not using the default port)."
			sb.append(':');
			sb.append(uri.getPort());
		}
		sb.append(HTTP_END_OF_LINE);

		// Origin
		// This field is added only if the client is a web browser (section 4.1, page)
		// This library is not intended to be used by web browsers so we don't add it

		// Upgrade
		sb.append("Upgrade: websocket");
		sb.append(HTTP_END_OF_LINE);

		// Connection
		sb.append("Connection: Upgrade");
		sb.append(HTTP_END_OF_LINE);

		// Key
		sb.append("Sec-WebSocket-Key: ");
		sb.append(nonce);
		sb.append(HTTP_END_OF_LINE);

		// Version
		sb.append("Sec-WebSocket-Version: ");
		sb.append(RFC_WEBSOCKET_VERSION);
		sb.append(HTTP_END_OF_LINE);

		// Extensions
		// TODO future versions may manage extensions (Sec-WebSocket-Extensions)

		// Sub-protocols
		// TODO futures version may manage sub-protocols (Sec-WebSocket-Protocol)

		// custom headers
		for (String header : this.customHeaders) {
			sb.append(header);
			sb.append(HTTP_END_OF_LINE);
		}

		// HTTP request is complete, place the ending marker
		sb.append(HTTP_END_OF_LINE);

		// Send this handshake
		Messages.LOGGER.log(Level.FINER, Messages.CATEGORY, Messages.SENDING, this, sb);
		os.write(sb.toString().getBytes());
		os.flush();
	}

	/**
	 * Section 4.1. Client Requirements gives a procedure to validate a server response to the opening handshake. The
	 * list is given at the end of page 19. Note that section 4.2.2. Sending the Server's Opening Handshake described
	 * how the response is supposed to be formed by the server.
	 *
	 * @param response
	 *            the HTTP response sent by the server
	 * @param nonce
	 *            nonce as described if the RFC6455 (random 16-byte value, as a base64 encoded string), not used in this
	 *            implementation.
	 * @return true if the response is valid; false otherwise
	 */
	private boolean validateResponse(HttpResponse response, String nonce) {

		// 1. Status code should be 101
		if (response.getStatusLine().getStatusCode() != 101) {
			// Server doesn't support websocket
			// TODO future versions may handle some codes specifically
			Messages.LOGGER.log(Level.WARNING, Messages.CATEGORY, Messages.INVALID_STATUS, this);
			return false;
		}

		// 2. There must be an Upgrade header field with value "websocket"
		Header upgrade = response.getHeader("Upgrade");
		if (upgrade == null || upgrade.getValue().compareToIgnoreCase("websocket") != 0) {
			Messages.LOGGER.log(Level.WARNING, Messages.CATEGORY, Messages.INVALID_UPGRADE, this);
			return false;
		}

		// 3. There must be a Connection header field with value "upgrade"
		Header connection = response.getHeader("Connection");
		if (connection == null || connection.getValue().compareToIgnoreCase("upgrade") != 0) {
			Messages.LOGGER.log(Level.WARNING, Messages.CATEGORY, Messages.INVALID_CONNECTION, this);
			return false;
		}

		// 4. There must be a Sec-WebSocket-Accept header field with a correct value
		Header secWebsocketAccept = response.getHeader("Sec-WebSocket-Accept");
		if (secWebsocketAccept == null) {
			Messages.LOGGER.log(Level.WARNING, Messages.CATEGORY, Messages.INVALID_SEC_WEBSOCKET_ACCEPT, this);
			return false;
		}
		// else {
		// Section 4.2.2. Sending the Server’s Opening Handshake, page 18, bullet 5
		// String expected = nonce + RFC_MAGIC_SERVER_KEY;
		// FIXME we lack SHA-1 algorithm to check if the value if correct
		// }

		// 5. Validate extensions given by Sec-WebSocket-Extensions header field
		// TODO future versions may manage extensions

		// 6. Validate sub-protocols given by Sec-WebSocket-Protocol header field
		// TODO future versions may manage sub-protocols

		// So far, so good
		return true;
	}
}
