/*
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.net.util.ssl;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;

import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509KeyManager;
import javax.net.ssl.X509TrustManager;

/**
 * Provides utility methods to create SSL contexts.
 * <p>
 * The following snippet shows how to create a SSL context for a client:
 *
 * <pre>
 * X509Certificate certificate = CertificatesHelper.getCertificate(CERTIFICATE_PATH);
 * X509TrustManager trustManager = SslContextFactory.createTrustManager(certificate);
 * SSLContext sslContext = SslContextFactory.createSslContext(SslProtocols.TLS, trustManager);
 * </pre>
 */
public class SslContextFactory {

	private SslContextFactory() {
		// private constructor
	}

	/**
	 * Creates a SSL context.
	 * <p>
	 * The {@link #createTrustManager} and {@link #createKeyManager} methods can be used to create the trust manager and
	 * the key managers easily.
	 *
	 * @param protocol
	 *            the protocol (see {@link SslProtocols}).
	 * @param trustManager
	 *            the trust manager.
	 * @param keyManagers
	 *            the key managers.
	 * @return the new SSL context.
	 */
	public static SSLContext createSslContext(String protocol, TrustManager trustManager, KeyManager... keyManagers) {
		SSLContext sslContext;
		try {
			sslContext = SSLContext.getInstance(protocol);
		} catch (NoSuchAlgorithmException e) {
			throw new IllegalArgumentException(e);
		}

		try {
			sslContext.init(keyManagers, new TrustManager[] { trustManager }, null);
		} catch (KeyManagementException e) {
			throw new UnsupportedOperationException(e);
		}

		return sslContext;
	}

	/**
	 * Creates a trust manager that stores the given certificates.
	 *
	 * @param certificates
	 *            the trusted certificates.
	 * @return the new trust manager.
	 */
	public static X509TrustManager createTrustManager(X509Certificate... certificates) {
		TrustManagerFactory trustManagerFactory;

		try {
			KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
			keyStore.load(null, null);

			int aliasId = 0;
			for (X509Certificate certificate : certificates) {
				assert (certificate != null);
				String alias = Integer.toString(aliasId++);
				keyStore.setCertificateEntry(alias, certificate);
			}

			trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
			trustManagerFactory.init(keyStore);
		} catch (GeneralSecurityException | IOException e) {
			throw new UnsupportedOperationException(e);
		}

		TrustManager[] trustManagers = trustManagerFactory.getTrustManagers();
		TrustManager trustManager = trustManagers[0];
		assert (trustManager != null);
		return (X509TrustManager) trustManager;
	}

	/**
	 * Creates a key manager that stores the given key and its associated certificate chain.
	 *
	 * @param key
	 *            the key.
	 * @param keyPassword
	 *            the password to decrypt the key.
	 * @param certificateChain
	 *            the certificate chain.
	 * @return the new key manager.
	 */
	public static X509KeyManager createKeyManager(byte[] key, char[] keyPassword, X509Certificate... certificateChain) {
		KeyManagerFactory keyManagerFactory;

		try {
			KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
			keyStore.load(null, null);
			keyStore.setKeyEntry("key", key, certificateChain);

			keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
			keyManagerFactory.init(keyStore, keyPassword);
		} catch (GeneralSecurityException | IOException e) {
			throw new UnsupportedOperationException(e);
		}

		KeyManager[] keyManagers = keyManagerFactory.getKeyManagers();
		KeyManager keyManager = keyManagers[0];
		assert (keyManager != null);
		return (X509KeyManager) keyManager;
	}
}
