/*
 * Java
 *
 * Copyright 2017-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.net.util.wifi;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.ecom.wifi.AccessPoint;
import ej.ecom.wifi.SecurityMode;

/**
 * A configuration to connect to an {@link AccessPoint}.
 */
public class AccessPointConfiguration {
	private String passphrase;

	private String ssid;

	private SecurityMode securityMode;

	private AccessPoint accessPoint;

	/**
	 * Instantiates an {@link AccessPointConfiguration} with a Service Set IDentification (SSID).
	 *
	 * @param ssid
	 *            the SSID to set.
	 * @throws NullPointerException
	 *             when ssid is <code>null</code>.
	 */
	public AccessPointConfiguration(@NonNull String ssid) throws NullPointerException {
		if (ssid == null) {
			throw new NullPointerException();
		}
		this.ssid = ssid;
	}

	/**
	 * Instantiates an {@link AccessPointConfiguration} with an access point.
	 *
	 * @param accessPoint
	 *            the AccessPoint to use.
	 * @throws NullPointerException
	 *             if accessPoint is <code>null</code> or its SSID.
	 *
	 */
	public AccessPointConfiguration(@NonNull AccessPoint accessPoint) throws NullPointerException {
		this(accessPoint.getSSID());
		this.accessPoint = accessPoint;
	}

	/**
	 * Instantiates an {@link AccessPointConfiguration}.
	 *
	 * @param ssid
	 *            the SSID to set.
	 * @param passphrase
	 *            the passphrase to use.
	 * @param securityMode
	 *            the securityMode to use.
	 * @throws NullPointerException
	 *             when ssid is <code>null</code>.
	 */
	public AccessPointConfiguration(@NonNull String ssid, @Nullable String passphrase,
			@Nullable SecurityMode securityMode) {
		this(ssid);
		this.passphrase = passphrase;
		this.securityMode = securityMode;
	}

	/**
	 * Gets the passphrase.
	 *
	 * @return the passphrase, can be <code>null</code>.
	 */
	public String getPassphrase() {
		return this.passphrase;
	}

	/**
	 * Sets the passphrase.
	 *
	 * @param passphrase
	 *            the passphrase to set.
	 */
	public void setPassphrase(@Nullable String passphrase) {
		this.passphrase = passphrase;
	}

	/**
	 * Gets the Service Set IDentification (SSID). If an {@link AccessPoint} is set, gets the {@link AccessPoint} SSID.
	 *
	 * @return the SSID.
	 * @see AccessPoint#getSSID()
	 */
	public String getSSID() {
		String ret = this.ssid;
		AccessPoint point = this.accessPoint;
		if (point != null) {
			ret = point.getSSID();
		}
		return ret;
	}

	/**
	 * Sets the Service Set IDentification (SSID). If an {@link AccessPoint} is set, sets the {@link AccessPoint} SSID.
	 *
	 * @param ssid
	 *            the SSID to set.
	 * @throws NullPointerException
	 *             if the SSID is <code>null</code>.
	 */
	public void setSSID(@NonNull String ssid) throws NullPointerException {
		if (ssid == null) {
			throw new NullPointerException();
		}
		this.ssid = ssid;
		AccessPoint point = this.accessPoint;
		if (point != null) {
			point.setSSID(ssid);
		}
	}

	/**
	 * Gets the security mode. If an {@link AccessPoint} is set, set the {@link AccessPoint} Security Mode
	 *
	 * @return the security mode.
	 * @see AccessPoint#getSecurityMode()
	 */
	public SecurityMode getSecurityMode() {
		SecurityMode ret = this.securityMode;
		AccessPoint point = this.accessPoint;
		if (point != null) {
			ret = point.getSecurityMode();
		}
		return ret;
	}

	/**
	 * Sets the securityMode. If an {@link AccessPoint} is set, sets the {@link AccessPoint} securityMode.
	 *
	 * @param securityMode
	 *            the securityMode to set.
	 */
	public void setSecurityMode(@Nullable SecurityMode securityMode) {
		this.securityMode = securityMode;
		AccessPoint point = this.accessPoint;
		if (point != null) {
			point.setSecurityMode(securityMode);
		}
	}

	/**
	 * Gets the accessPoint.
	 *
	 * @return the accessPoint.
	 */
	public AccessPoint getAccessPoint() {
		return this.accessPoint;
	}

	/**
	 * Sets the accessPoint.
	 *
	 * @param accessPoint
	 *            the accessPoint to set.
	 */
	public void setAccessPoint(@Nullable AccessPoint accessPoint) {
		this.accessPoint = accessPoint;
	}
}
