/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.net;

import com.microej.kf.util.module.SandboxedModule;
import ej.annotation.Nullable;
import ej.basictool.ArrayTools;
import ej.basictool.map.PackedMap;
import ej.kf.Kernel;

import java.net.InetAddress;
import java.net.Socket;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * The network resources controller.
 *
 * <p>
 * It controls the numbers of opened connections and the network bandwidth. By default, there are no limits on the
 * network resources usage.
 *
 * <p>
 * When an application tries to perform a network operation (open/read/write), the controller checks if a network limit
 * setting is defined for the application and denies the operation if its execution will lead to exceeding one of the
 * limits.
 */
public class NetResourcesController {

	private int maxConnections;
	private final PackedMap<Subnet, Bandwidth> maxBandwidths;
	private Socket[] currentOpenedConnections;
	private final Logger logger;

	/**
	 * Instantiates a network resource controller.
	 */
	public NetResourcesController() {
		this.logger = Logger.getLogger(NetResourcesController.class.getName());
		this.maxBandwidths = new PackedMap<>();
		reset();
	}

	/**
	 * Resets the controller.
	 */
	public synchronized void reset() {
		this.maxConnections = -1;
		this.currentOpenedConnections = new Socket[0];
		synchronized (this.maxBandwidths) {
			this.maxBandwidths.clear();
		}
	}

	/////////////////////////////////////////
	// Open socket action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the open socket action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed.
	 *
	 * @param module
	 * 		the module which performs the action.
	 * @param openSocket
	 * 		the open socket action.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, OpenSocket openSocket) throws SecurityException {
		if (this.maxConnections < 0 || this.currentOpenedConnections.length < this.maxConnections) {
			return;
		}

		if (this.logger.isLoggable(Level.FINE)) {
			this.logger.fine("Opened connections limit reached: only " + this.maxConnections // $NON-NLS-1$
					+ " connections are allowed to be opened)"); // $NON-NLS-1$
		}
		throw new OpenedSocketsLimitException();
	}

	/**
	 * Called when the execution of the given open socket action is finished.
	 *
	 * @param module
	 * 		the module which performs the given open socket action.
	 * @param openSocket
	 * 		the open socket action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, OpenSocket openSocket, boolean withSuccess) {
		if (withSuccess) {
			Kernel.enter();
			this.currentOpenedConnections = ArrayTools.add(this.currentOpenedConnections, openSocket.getSocket());
		}
	}

	/////////////////////////////////////////
	// Close socket action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the close socket action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed.
	 *
	 * @param module
	 * 		the module which performs the action.
	 * @param closeSocket
	 * 		the close socket action.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, CloseSocket closeSocket) throws SecurityException {
		// Called by closeSocket.onStart().
	}

	/**
	 * Called when the execution of the given close socket action is finished.
	 *
	 * @param module
	 * 		the module which performs the given close socket action.
	 * @param closeSocket
	 * 		the close socket action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, CloseSocket closeSocket, boolean withSuccess) {
		if (withSuccess) {
			Kernel.enter();
			this.currentOpenedConnections = ArrayTools.remove(this.currentOpenedConnections, closeSocket.getSocket());
		}
	}

	/////////////////////////////////////////
	// Read socket action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the read socket action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed and throttles (delays) it if needed.
	 * <p>
	 *
	 * @param module
	 * 		the module that performs the action.
	 * @param readSocket
	 * 		the read socket action.
	 * @return the number of bytes read.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized int onStart(SandboxedModule module, ReadSocket readSocket) throws SecurityException {
		InetAddress remoteAddress = readSocket.getSocket().getInetAddress();
		Bandwidth bandwidth = getMaxBandwidth(remoteAddress);
		if (bandwidth != null) {
			return (int) bandwidth.getReadBandwidthChecker().check(readSocket.getNbBytes(), false);
		}
		return readSocket.getNbBytes();
	}

	/**
	 * Called when the execution of the given read socket action is finished.
	 *
	 * @param module
	 * 		the module which performs the given read socket action.
	 * @param readSocket
	 * 		the read socket action.
	 * @param nbBytesRead
	 * 		number of bytes actually read.
	 */
	public synchronized void onEnd(SandboxedModule module, ReadSocket readSocket, int nbBytesRead) {
		if (nbBytesRead > 0) {
			InetAddress remoteAddress = readSocket.getSocket().getInetAddress();
			Bandwidth bandwidth = getMaxBandwidth(remoteAddress);
			if (bandwidth != null) {
				bandwidth.getReadBandwidthChecker().update(nbBytesRead);
			}
		}
	}

	/////////////////////////////////////////
	// Write socket action
	/////////////////////////////////////////

	/**
	 * Called when the execution of the write socket action is about to start.
	 * <p>
	 * It checks if the action is allowed to be executed and throttles (delays) it if needed.
	 * <p>
	 *
	 * @param module
	 * 		the module that performs the action.
	 * @param writeSocket
	 * 		the write socket action.
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public synchronized void onStart(SandboxedModule module, WriteSocket writeSocket) throws SecurityException {
		InetAddress remoteAddress = writeSocket.getSocket().getInetAddress();
		Bandwidth maxBandwidth = getMaxBandwidth(remoteAddress);
		if (maxBandwidth != null) {
			maxBandwidth.getWriteBandwidthChecker().check(writeSocket.getNbBytes(), true);
		}
	}

	/**
	 * Called when the execution of the given write socket action is finished.
	 *
	 * @param module
	 * 		the module which performs the given write socket action.
	 * @param writeSocket
	 * 		the write socket action.
	 * @param withSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public synchronized void onEnd(SandboxedModule module, WriteSocket writeSocket, boolean withSuccess) {
		if (withSuccess) {
			InetAddress remoteAddress = writeSocket.getSocket().getInetAddress();
			Bandwidth maxBandwidth = getMaxBandwidth(remoteAddress);
			if (maxBandwidth != null) {
				maxBandwidth.getWriteBandwidthChecker().update(writeSocket.getNbBytes());
			}
		}
	}

	/**
	 * Gets the bandwidth for which the associated subnet matches the given ip address.
	 *
	 * @param ipAddress
	 * 		the ip address.
	 * @return the bandwidth for which the associated subnet matches the given ip address; or null if no bandwidth
	 * 		found.
	 */
	@Nullable
	private Bandwidth getMaxBandwidth(@Nullable InetAddress ipAddress) {
		if (ipAddress == null) {
			return null;
		}
		synchronized (this.maxBandwidths) {
			for (Subnet subnet : this.maxBandwidths.keySet()) {
				if (subnet.matches(ipAddress)) {
					return this.maxBandwidths.get(subnet);
				}
			}
		}
		return null;
	}

	/**
	 * Sets the maximum bandwidth for the given subnet.
	 *
	 * <p>
	 * Note that if a maximum bandwidth is already assigned for the subnet, this new maximum bandwidth will override the
	 * previous one.
	 *
	 * @param subnet
	 * 		the subnet for which the bandwidth is set.
	 * @param bandwidth
	 * 		the maximum bandwidth to set.
	 */
	public void setMaxBandwidth(Subnet subnet, Bandwidth bandwidth) {
		synchronized (this.maxBandwidths) {
			this.maxBandwidths.put(subnet, bandwidth);
		}
	}

	/**
	 * Sets the maximum number of network connections allowed to be opened.
	 *
	 * @param maxOpenedConnections
	 * 		the maximum number of network connections allowed. A negative value means there is no limit.
	 */
	public void setMaxOpenedConnections(int maxOpenedConnections) {
		this.maxConnections = maxOpenedConnections;
	}

	/**
	 * Returns the current opened connections.
	 *
	 * @return the current opened connections.
	 */
	public synchronized Socket[] getOpenedConnections() {
		return this.currentOpenedConnections.clone();
	}

	/**
	 * @return the maximum number of network connections allowed. A negative value means there is no limit.
	 */
	public int getMaxOpenedConnections() {
		return this.maxConnections;
	}

	/**
	 * @param subnet
	 * 		the subnet for which the bandwidth was set.
	 * @return the maximum network bandwidth authorized for this module.
	 */
	@Nullable
	public Bandwidth getNetworkMaxBandwidth(Subnet subnet) {
		synchronized (this.maxBandwidths) {
			return this.maxBandwidths.get(subnet);
		}
	}
}
