/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display.buffer;

import ej.fp.Image;
import ej.fp.Widget;
import ej.fp.widget.display.DisplayBufferManager;
import ej.microui.display.Rectangle;

/**
 * The display buffer policy is responsible of managing the different buffers.
 */
public interface DisplayBufferPolicy {

	/**
	 * Sets the display properties.
	 *
	 * @param displayWidget
	 *            the display widget.
	 * @param width
	 *            the width of the display.
	 * @param height
	 *            the height of the display
	 * @param initialColor
	 *            the initial color of the display.
	 */
	void setDisplayProperties(Widget displayWidget, int width, int height, int initialColor);

	/**
	 * Performs a flush: the content of the back buffer has to be "transmitted" to the display front buffer.
	 * <p>
	 * The content of given rectangles in the graphics buffer (back buffer) must be displayed into the LCD buffer (front
	 * buffer). The rectangles specified are the regions which have changed during last draw and only these regions
	 * should be updated (dirty regions). The number of rectangles depends on the display buffer refresh strategy (BRS).
	 *
	 * @param bufferManager
	 *            the buffer manager.
	 * @param rectangles
	 *            the modified rectangular regions.
	 */
	void flush(DisplayBufferManager bufferManager, Rectangle[] rectangles);

	/**
	 * Gets the number of buffers.
	 *
	 * @return the number of buffers.
	 */
	int getBufferCount();

	/**
	 * Gets whether or not there are several buffers for the display.
	 * <p>
	 * In other words, if the drawings are done in a buffer different than the one used by the display or not.
	 *
	 * @return <code>true</code> if there is more than one buffer, <code>false</code> otherwise.
	 */
	public boolean isDoubleBuffered();

	/**
	 * Gets the back buffer.
	 *
	 * @return the back buffer.
	 */
	Image getBackBuffer();

	/**
	 * Gets the front buffer.
	 *
	 * @return the front buffer.
	 */
	Image getFrontBuffer();

	/**
	 * Disposes the buffers.
	 */
	void dispose();

}
