/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display.brs;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import ej.fp.Image;
import ej.fp.widget.display.DisplayBufferManager;
import ej.microui.display.Rectangle;

/**
 * The "predraw" strategy consists in restoring the back buffer with the content of the previous drawings (the past)
 * just before the very first drawing after a flush (and not just after the flush). A partial restore is only performed
 * if the first drawing uses a different back buffer to the previous drawings (before the flush) and which not fits the
 * full display region. Otherwise, the restore is useless (the drawings will reuse the same back buffer or the first
 * drawing will fully erase the past).
 */
public class PredrawRefreshStrategy implements BufferRefreshStrategy {

	private int displayWidth;
	private int displayHeight;

	// List of dirty rectangles in the other buffers than the current one.
	private List<Rectangle>[] dirtyRectangles;
	private int currentDirtyRectangle;
	// List of dirty rectangles since previous flush.
	private List<Rectangle> currentDirtyRectangles;
	private Image previousBackBuffer;

	private boolean hasFlushed;

	@Override
	public void setDisplaySize(int width, int height) {
		this.displayWidth = width;
		this.displayHeight = height;
	}

	@Override
	public void setBufferCount(int bufferCount) {
		int count = Math.max(bufferCount - 1, 1);
		this.dirtyRectangles = new List[count];
		for (int i = 0; i < count; i++) {
			this.dirtyRectangles[i] = new ArrayList<>();
		}
		this.currentDirtyRectangles = new ArrayList<>();
	}

	@Override
	public void newDrawingRegion(DisplayBufferManager bufferManager, Rectangle rectangle, boolean drawingNow) {
		if (drawingNow) {
			Image currentBackBuffer = bufferManager.getCurrentBackBuffer();
			synchronized (bufferManager.getMonitor()) {
				if (this.hasFlushed) {
					this.hasFlushed = false;
					if (this.previousBackBuffer != currentBackBuffer) {
						List<Rectangle> dirtyRectanglesBuffer = this.dirtyRectangles[this.currentDirtyRectangle];
						if (isFullSize(rectangle)) {
							// Nothing to restore.
						} else {
							cleanDirtyRectangles(rectangle, dirtyRectanglesBuffer);
							// Restore remaining dirty regions.
							for (Rectangle dirtyRectangle : dirtyRectanglesBuffer) {
								bufferManager.restore(dirtyRectangle);
							}
						}
						dirtyRectanglesBuffer.clear();
						this.currentDirtyRectangle = (this.currentDirtyRectangle + 1) % (this.dirtyRectangles.length);
					}
					this.currentDirtyRectangles.clear();
				}
				add(this.currentDirtyRectangles, rectangle);
				addRectangleToOtherBuffers(rectangle);
			}
			this.previousBackBuffer = currentBackBuffer;
		} else {
			if (!this.hasFlushed) {
				List<Rectangle> dirtyRectanglesBuffer = this.dirtyRectangles[this.currentDirtyRectangle];
				cleanDirtyRectangles(rectangle, dirtyRectanglesBuffer);
			}
			addRectangleToOtherBuffers(rectangle);
		}
	}

	private void cleanDirtyRectangles(Rectangle drawnRectangle, List<Rectangle> dirtyRectanglesBuffer) {
		// Remove dirty rectangles included in the next drawing.
		List<Rectangle> parts = new ArrayList<>();
		for (Iterator<Rectangle> iterator = dirtyRectanglesBuffer.iterator(); iterator.hasNext();) {
			Rectangle candidate = iterator.next();
			if (drawnRectangle.contains(candidate)) {
				iterator.remove();
			} else if (candidate.contains(drawnRectangle)) {
				int x1 = drawnRectangle.getX1();
				int x2 = drawnRectangle.getX2();
				int y1 = drawnRectangle.getY1();
				int y2 = drawnRectangle.getY2();
				// Maybe add a surface condition? (for example if the last clip is very small.)
				iterator.remove();
				// Cut the bigger zone into several parts.
				if (candidate.getY1() != y1) { // Top.
					parts.add(new Rectangle(candidate.getX1(), candidate.getY1(), candidate.getX2(), y1 - 1));
				}
				if (candidate.getY2() != y2) {// Bottom.
					parts.add(new Rectangle(candidate.getX1(), y2 + 1, candidate.getX2(), candidate.getY2()));
				}
				if (candidate.getX1() != x1) {// Left.
					parts.add(new Rectangle(candidate.getX1(), y1, x1 - 1, y2));
				}
				if (candidate.getX2() != x2) {// Right.
					parts.add(new Rectangle(x2 + 1, y1, candidate.getX2(), y2));
				}
			}
		}
		dirtyRectanglesBuffer.addAll(parts);
	}

	private void addRectangleToOtherBuffers(Rectangle rectangle) {
		for (int i = 0; i < this.dirtyRectangles.length; i++) {
			List<Rectangle> dirtyRectanglesBuffer = this.dirtyRectangles[i];
			add(dirtyRectanglesBuffer, rectangle);
		}
	}

	private void add(List<Rectangle> rectangles, Rectangle rectangle) {
		if (!rectangles.isEmpty()) {
			Rectangle lastRectangle = rectangles.get(rectangles.size() - 1);
			if (lastRectangle.contains(rectangle)) {
				return;
			} else if (rectangle.contains(lastRectangle)) {
				// Replace with the bigger rectangle (only useful when the number of buffers is bigger than 2).
				rectangles.remove(rectangles.size() - 1);
			}
		}
		rectangles.add(rectangle);
	}

	@Override
	public void refresh(DisplayBufferManager bufferManager) {
		List<Rectangle> dirtyRectangles = new ArrayList<>(this.currentDirtyRectangles);
		bufferManager.flush(dirtyRectangles.toArray(new Rectangle[dirtyRectangles.size()]));
		this.hasFlushed = true;
	}

	private boolean isFullSize(Rectangle rectangle) {
		return rectangle.getX1() == 0 && rectangle.getY1() == 0 && rectangle.getX2() == this.displayWidth - 1
				&& rectangle.getY2() == this.displayHeight - 1;
	}

}
