/*
 * Java
 *
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.security;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import org.json.me.JSONArray;
import org.json.me.JSONException;
import org.json.me.JSONObject;
import org.json.me.JSONTokener;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;

/**
 * The JSON implementation of the policy file parser
 */
public class JsonSecurityPolicyLoader implements SecurityPolicyResourceLoader {

	private static final Logger LOGGER = Logger.getLogger(JsonSecurityPolicyLoader.class.getName());

	private @Nullable List<FeaturePolicyPermission> loadResourceFile(@Nullable InputStream inputStream) {

		final List<FeaturePolicyPermission> permissionList = new ArrayList<>();

		if (inputStream == null) {
			LOGGER.fine("An installed feature came without security policy file");
			return null;
		}

		try {

			JSONTokener jsonTokener = new JSONTokener(inputStream);

			JSONObject rootPermissionObject = new JSONObject(jsonTokener).getJSONObject("permissions");

			for (String permissionClassName : rootPermissionObject.keySet()) {

				// EXPECTED: loop class names
				if (permissionClassName != null) {

					// class names object
					JSONObject permissionObject = rootPermissionObject.getJSONObject(permissionClassName);

					PackedMap<String, List<String>> nameActionsMap = new PackedMap<>();

					populateActions(permissionObject, nameActionsMap);

					FeaturePolicyPermission policyPermission = new FeaturePolicyPermission(permissionClassName,
							nameActionsMap);
					addToList(permissionList, policyPermission);

				}

			}
		} catch (JSONException e) {
			LOGGER.severe("Error while parsing the policy security file: " + e.getMessage());
			return null;
		}

		return permissionList;
	}

	private void populateActions(JSONObject permissionObject, PackedMap<String, List<String>> nameActionsMap)
			throws JSONException {
		for (String param : permissionObject.keySet()) {
			if (param != null) {
				List<String> actionList = new ArrayList<>();
				JSONArray actionArray = permissionObject.getJSONArray(param);
				Iterator<Object> actionIterator = actionArray.iterator();

				while (actionIterator.hasNext()) {
					String action = (String) actionIterator.next();
					actionList.add(action);
				}

				nameActionsMap.put(param, actionList);
			}
		}
	}

	private void addToList(List<FeaturePolicyPermission> list, FeaturePolicyPermission policyPermission) {

		List<FeaturePolicyPermission> copy = new ArrayList<>(list);

		if (copy.isEmpty()) {
			list.add(policyPermission);
			return;
		}

		boolean hasAlreadyBeenRegistered = false;
		for (FeaturePolicyPermission registeredPermission : copy) {
			if (registeredPermission.getPermissionClassName().equals(policyPermission.getPermissionClassName())) {
				// do nothing or override ?
				hasAlreadyBeenRegistered = true;
				break;
			}
		}

		if (!hasAlreadyBeenRegistered) {
			list.add(policyPermission);
		}
	}

	@Override
	public @Nullable List<FeaturePolicyPermission> loadFeaturePermissions(@Nullable InputStream inputStream) {

		return loadResourceFile(inputStream);
	}

}
