/*
 * Java
 *
 * Copyright 2020-2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.drawing.LLDWPainter.DrawingCap;
import ej.drawing.LLDWPainter.DrawingFlip;
import ej.microui.display.LLUIPainter.MicroUIGraphicsContext;
import ej.microui.display.LLUIPainter.MicroUIImage;
import ej.microui.display.LLUIPainter.MicroUIImageFormat;

/**
 * This interface provides all drawing methods useful to implement MicroUI drawing native methods.
 * <p>
 * This interface is implemented by the MicroUI graphical engine's internal software drawer. This drawer implements all
 * drawings in software. The rendering is exactly the same than the embedded algorithms listed in
 * <code>ui_drawing_soft.h</code>. See {@link LLUIDisplay#getUIDrawerSoftware()}.
 * <p>
 * This interface can be fully implemented by a dedicated platform class in platform front panel project. This allows to
 * implement drawing algorithms like embedded platform.
 */
public interface UIDrawing {

	/**
	 * Gets the supported buffered image format.
	 *
	 * @return the supported buffered image format
	 */
	MicroUIImageFormat handledFormat();

	/**
	 * Draws a pixel at given position.
	 *
	 * @param gc
	 *            drawing destination target.
	 * @param x
	 *            the pixel X coordinate.
	 * @param y
	 *            the pixel Y coordinate.
	 */
	default void writePixel(MicroUIGraphicsContext gc, int x, int y) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a line at between points x1,y1 (included) and x2,y2 (included). Note: x1 may be higher than x2 (and/or y1
	 * may be higher than y2).
	 *
	 * @param gc
	 *            drawing destination target.
	 * @param x1
	 *            the first pixel line X coordinate.
	 * @param y1
	 *            the first pixel line Y coordinate.
	 * @param x2
	 *            the last pixel line X coordinate.
	 * @param y2
	 *            the last pixel line Y coordinate.
	 */
	default void drawLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a horizontal line at between points x1,y (included) and x2,y (included). Caller ensures that x1 <= x2.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x1
	 *            the first pixel line X coordinate.
	 * @param x2
	 *            the last pixel line X coordinate.
	 * @param y
	 *            the both pixels line Y coordinate.
	 */
	default void drawHorizontalLine(MicroUIGraphicsContext gc, int x1, int x2, int y) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a vertical line at between points x,y1 (included) and x,y2 (included). Caller ensures that y1 <= y2.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the both pixels line X coordinate.
	 * @param y1
	 *            the first pixel line Y coordinate.
	 * @param y2
	 *            the last pixel line Y coordinate.
	 */
	default void drawVerticalLine(MicroUIGraphicsContext gc, int x, int y1, int y2) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a rectangle at from top-left point x1,y1 (included) and bottom-right point x2,y2 (included). Caller ensures
	 * that x1 <= x2 and y1 <= y2.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x1
	 *            the top-left pixel X coordinate.
	 * @param y1
	 *            the top-left pixel Y coordinate.
	 * @param x2
	 *            the bottom-right pixel X coordinate.
	 * @param y2
	 *            the top-right pixel Y coordinate.
	 */
	default void drawRectangle(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Fills a rectangle at from top-left point x1,y1 (included) and bottom-right point x2,y2 (included). Caller ensures
	 * that x1 <= x2 and y1 <= y2.
	 *
	 * @param gc
	 *            drawing destination target.
	 * @param x1
	 *            the top-left pixel X coordinate.
	 * @param y1
	 *            the top-left pixel Y coordinate.
	 * @param x2
	 *            the bottom-right pixel X coordinate.
	 * @param y2
	 *            the top-right pixel Y coordinate.
	 */
	default void fillRectangle(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a rounded rectangle at from top-left point x,y (included) and bottom-right point x+width-1,y+height-1
	 * (included).
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	default void drawRoundedRectangle(MicroUIGraphicsContext gc, int x, int y, int width, int height, int arcWidth,
			int arcHeight) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Fills a rounded rectangle at from top-left point x,y (included) and bottom-right point x+width-1,y+height-1
	 * (included).
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	default void fillRoundedRectangle(MicroUIGraphicsContext gc, int x, int y, int width, int height, int arcWidth,
			int arcHeight) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a circular arc covering the square defined by top-left point x,y (included) and bottom-right point
	 * x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given diameter is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	default void drawCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle,
			float arcAngle) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an elliptical arc covering the rectangle defined by top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included) .
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * rectangle whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squared rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	default void drawEllipseArc(MicroUIGraphicsContext gc, int x, int y, int width, int height, float startAngle,
			float arcAngle) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Fills a circular arc covering the square defined by top-left point x,y (included) and bottom-right point
	 * x+diameter-1,y+diameter-1 (included).
	 *
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 *
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 *
	 * If either the given diameter is negative or zero, or if arcAngle is zero, nothing is drawn.
	 *
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	default void fillCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle,
			float arcAngle) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Fills an arc covering the rectangle defined by top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * rectangle whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squared rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	default void fillEllipseArc(MicroUIGraphicsContext gc, int x, int y, int width, int height, float startAngle,
			float arcAngle) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an ellipse covering the rectangle defined by top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included).
	 * <p>
	 * If either the given width or height is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the ellipse width.
	 * @param height
	 *            the ellipse height.
	 */
	default void drawEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Fills an ellipse covering the rectangle defined by top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included).
	 * <p>
	 * If either the given width or height is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the ellipse width.
	 * @param height
	 *            the ellipse height.
	 */
	default void fillEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a circle covering the square defined by top-left point x,y (included) and bottom-right point
	 * x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * If the given diameter is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the circle square size.
	 */
	default void drawCircle(MicroUIGraphicsContext gc, int x, int y, int diameter) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Fills a circle covering the square defined by top-left point x,y (included) and bottom-right point
	 * x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * If the given diameter is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the circle square size.
	 */
	default void fillCircle(MicroUIGraphicsContext gc, int x, int y, int diameter) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a region of an image. The region of the image to draw is given relative to the image (origin at the
	 * upper-left corner) as a rectangle.
	 * <p>
	 * If the specified source region exceeds the image bounds, the copied region is limited to the image boundary. If
	 * the copied region goes out of the bounds of the graphics context area, pixels out of the range will not be drawn.
	 * <p>
	 * A global opacity value is given. When this value is 0xff (255, opaque), that means the image is drawn on the
	 * graphics context without managing an extra opacity. Only the image transparent pixels must have been merged with
	 * destination. All image opaque pixels override destination.
	 * <p>
	 * When this value is a value between 0 and 0xff, that means each pixel of the image must be merged with destination
	 * in addition with the image transparent pixels. An image opaque pixel becomes transparent (its opacity is the
	 * given alpha) and the opacity of an image transparent pixel becomes (alpha * alpha(pixel)) / 255.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param regionY
	 *            the y coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawImage(MicroUIGraphicsContext gc, MicroUIImage img, int regionX, int regionY, int width, int height,
			int x, int y, int alpha) {
		LLUIDisplay.Instance.getUIImageDrawer(img).draw(gc, img, regionX, regionY, width, height, x, y, alpha);
	}

	/**
	 * Draws a thick point with fade at given position.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the point X coordinate.
	 * @param y
	 *            the point Y coordinate.
	 * @param thickness
	 *            the point thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	default void drawThickFadedPoint(MicroUIGraphicsContext gc, int x, int y, int thickness, int fade) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a thick line with fade between given points.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x1
	 *            the x coordinate of the start of the line
	 * @param y1
	 *            the y coordinate of the start of the line
	 * @param x2
	 *            the x coordinate of the end of the line
	 * @param y2
	 *            the y coordinate of the end of the line
	 * @param thickness
	 *            the line thickness.
	 * @param fade
	 *            the fade to apply.
	 * @param start
	 *            cap representation of start of shape
	 * @param end
	 *            cap representation of end of shape
	 */
	default void drawThickFadedLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2, int thickness, int fade,
			DrawingCap start, DrawingCap end) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a thick circle with fade covering the square specified by its diameter. If diameter is negative or zero,
	 * nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the circle is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the circle is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param thickness
	 *            the circle thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	default void drawThickFadedCircle(MicroUIGraphicsContext gc, int x, int y, int diameter, int thickness, int fade) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a thick circle with fade arc covering the specified square. The arc is drawn from startAngle up to arcAngle
	 * degrees.
	 * <p>
	 * The center of the arc is defined as the center of the square whose origin is at (x,y) (upper-left corner) and
	 * whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If diameter is negative or zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square. Thus for a non squared square angles are
	 * skewed along either height or width.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the arc is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 * @param thickness
	 *            the arc thickness.
	 * @param fade
	 *            the fade to apply.
	 * @param start
	 *            cap representation of start of shape
	 * @param end
	 *            cap representation of end of shape
	 */
	default void drawThickFadedCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle,
			float arcAngle, int thickness, int fade, DrawingCap start, DrawingCap end) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a thick ellipse with fade covering the specified rectangle.
	 * <p>
	 * The center of the ellipse is defined as the center of the rectangle whose origin is at (x,y) (upper-left corner)
	 * and whose dimension is given by width and height.
	 * <p>
	 * If either width or height is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param width
	 *            the width of the ellipse to draw
	 * @param height
	 *            the height of the ellipse to draw
	 * @param thickness
	 *            the ellipse thickness.
	 * @param fade
	 *            the fade to apply.
	 */
	default void drawThickFadedEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height, int thickness,
			int fade) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a thick line between given points.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x1
	 *            the x coordinate of the start of the line
	 * @param y1
	 *            the y coordinate of the start of the line
	 * @param x2
	 *            the x coordinate of the end of the line
	 * @param y2
	 *            the y coordinate of the end of the line
	 * @param thickness
	 *            the line thickness.
	 */
	default void drawThickLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2, int thickness) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a thick circle covering the square specified by its diameter.
	 * <p>
	 * If diameter is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the circle is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the circle is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param thickness
	 *            the circle thickness.
	 */
	default void drawThickCircle(MicroUIGraphicsContext gc, int x, int y, int diameter, int thickness) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a thick ellipse covering the specified rectangle.
	 * <p>
	 * The center of the ellipse is defined as the center of the rectangle whose origin is at (x,y) (upper-left corner)
	 * and whose dimension is given by width and height.
	 * <p>
	 * If either width or height is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the rectangle where the ellipse is drawn
	 * @param width
	 *            the width of the ellipse to draw
	 * @param height
	 *            the height of the ellipse to draw
	 * @param thickness
	 *            the ellipse thickness.
	 */
	default void drawThickEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height, int thickness) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws a thick arc covering the square specified by its diameter.
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If diameter is negative, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the x coordinate of the upper-left corner of the square where the arc is drawn
	 * @param y
	 *            the y coordinate of the upper-left corner of the square where the arc is drawn
	 * @param diameter
	 *            the diameter of the circle to draw
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 * @param thickness
	 *            the arc thickness.
	 */
	default void drawThickCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle,
			float arcAngle, int thickness) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an image applying a flip (0, 90, 180 or 270 degrees with or without mirror).
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to draw.
	 * @param regionY
	 *            the y coordinate of the upper-left corner of the region to draw.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param flip
	 *            the flip to apply.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawFlippedImage(MicroUIGraphicsContext gc, MicroUIImage img, int regionX, int regionY, int width,
			int height, int x, int y, DrawingFlip flip, int alpha) {
		LLUIDisplay.Instance.getUIImageDrawer(img).drawFlipped(gc, img, regionX, regionY, width, height, x, y, flip,
				alpha);
	}

	/**
	 * Draws an image applying a free rotation (0 to 360 degrees).
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative where the graphical object will be drawn.
	 * <p>
	 * This method uses the nearest neighbor algorithm to render the content. This algorithm is faster than bilinear
	 * algorithm but its rendering is faster.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param rotationX
	 *            the x coordinate of the rotation center.
	 * @param rotationY
	 *            the y coordinate of the rotation center.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawRotatedImageNearestNeighbor(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y,
			int rotationX, int rotationY, float angle, int alpha) {
		LLUIDisplay.Instance.getUIImageDrawer(img).drawRotatedNearestNeighbor(gc, img, x, y, rotationX, rotationY,
				angle, alpha);
	}

	/**
	 * Draws an image applying a free rotation (0 to 360 degrees).
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative where the graphical object will be drawn.
	 * <p>
	 * This method uses the bilinear algorithm to render the content. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param rotationX
	 *            the x coordinate of the rotation center.
	 * @param rotationY
	 *            the y coordinate of the rotation center.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawRotatedImageBilinear(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, int rotationX,
			int rotationY, float angle, int alpha) {
		LLUIDisplay.Instance.getUIImageDrawer(img).drawRotatedBilinear(gc, img, x, y, rotationX, rotationY, angle,
				alpha);
	}

	/**
	 * Draws an image applying a scaling.
	 * <p>
	 * This method uses the nearest neighbor algorithm to render the content. This algorithm is faster than bilinear
	 * algorithm but its rendering is faster.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param factorX
	 *            scaling X factor.
	 * @param factorY
	 *            scaling Y factor.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawScaledImageNearestNeighbor(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y,
			float factorX, float factorY, int alpha) {
		LLUIDisplay.Instance.getUIImageDrawer(img).drawScaledNearestNeighbor(gc, img, x, y, factorX, factorY, alpha);
	}

	/**
	 * Draws an image applying a scaling.
	 * <p>
	 * This method uses the bilinear algorithm to render the content. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param factorX
	 *            scaling X factor.
	 * @param factorY
	 *            scaling Y factor.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawScaledImageBilinear(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, float factorX,
			float factorY, int alpha) {
		LLUIDisplay.Instance.getUIImageDrawer(img).drawScaledBilinear(gc, img, x, y, factorX, factorY, alpha);
	}

}
