/*
 * Copyright 2016-2019 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.service.loader;

import ej.annotation.NonNull;
import ej.service.ServiceLoadingException;

/**
 * Helper class for service loaders.
 */
public class ServiceLoaderHelper {

	private ServiceLoaderHelper() {
		// utility class
	}

	/**
	 * Gets an instance of a class, based on its name.
	 *
	 * @param parent
	 *            the parent class.
	 * @param className
	 *            the name of the class to instantiate.
	 * @param <T>
	 *            the class type.
	 * @return an instance of the given class.
	 * @throws ServiceLoadingException
	 *             if an error occurs while instantiating the service implementation.
	 */
	public static <T> T createClassInstance(Class<T> parent, String className) throws ServiceLoadingException {
		try {
			@SuppressWarnings("null") // The forName method cannot return null (otherwise it throws an exception).
			@NonNull
			Class<?> clazz = Class.forName(className);
			// ensure that is is the right type
			if (!parent.isAssignableFrom(clazz)) {
				throw new ClassCastException();
			}
			@SuppressWarnings("unchecked")
			Class<T> clazzTyped = (Class<T>) clazz;
			return createClassInstance(parent, clazzTyped);
		} catch (ClassNotFoundException | ClassCastException e) {
			// can't find or instantiate implementation
			@SuppressWarnings("null") // Class name cannot be null for sure.
			@NonNull
			String parentName = parent.getName();
			throw new ServiceLoadingException(parentName, e);
		}
	}

	/**
	 * Gets an instance of a class.
	 *
	 * @param parent
	 *            the parent class.
	 * @param clazz
	 *            the class to instantiate.
	 * @param <T>
	 *            the class type.
	 * @return an instance of the given class.
	 * @throws ServiceLoadingException
	 *             if an error occurs while instantiating the service implementation.
	 */
	public static <T> T createClassInstance(Class<T> parent, Class<? extends T> clazz) throws ServiceLoadingException {
		try {
			@SuppressWarnings("null") // The newInstance method cannot return null (otherwise it throws an exception).
			@NonNull
			T implementation = clazz.newInstance(); // cast ensured by type
			return implementation;
		} catch (InstantiationException | IllegalAccessException e) {
			// can't find or instantiate implementation
			@SuppressWarnings("null") // Class name cannot be null for sure.
			@NonNull
			String parentName = parent.getName();
			throw new ServiceLoadingException(parentName, e);
		}
	}

}
