/*
 * Copyright 2013-2019 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.service.loader;

import ej.annotation.NonNull;
import ej.annotation.Nullable;

/**
 * Service loader implementation that creates service instance from a declared implementation class name.
 * <p>
 * Subclasses need to provide the implementation class name matching the requested services. The implementation is then
 * instantiated using {@link Class#forName(String)} and {@link Class#newInstance()}. The singletons are stored in a map.
 */
public abstract class DependencyInjectionServiceLoader extends SimpleServiceLoader {

	@Override
	@Nullable
	protected <T> T createAlternativeImplementation(Class<T> service) {
		@SuppressWarnings("null") // Class name is not null for sure.
		@NonNull
		String serviceName = service.getName();
		String implementationName = getImplementationName(serviceName);
		if (implementationName != null) {
			T serviceImplementation = ServiceLoaderHelper.createClassInstance(service, implementationName);
			putService(service, serviceImplementation);
			return serviceImplementation;
		}
		return null;
	}

	/**
	 * Gets the implementation class name for the given service name.
	 *
	 * @param serviceName
	 *            the service name.
	 * @return the implementation class name, or {@code null} if none.
	 */
	@Nullable
	protected abstract String getImplementationName(String serviceName);

}