/*******************************************************************************
 * Copyright (c) 2009, 2014 IBM Corp.
 * Copyright 2019-2021 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *    http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Dave Locke - initial API and implementation and/or initial documentation
 *    Ian Craggs - ack control (bug 472172)
 *    MicroEJ Corp. - MicroPaho implementation and optimizations on MicroEJ
 */
package org.eclipse.paho.client.mqttv3;

/**
 * An MQTT message holds the application payload and options specifying how the message is to be delivered The message
 * includes a "payload" (the body of the message) represented as a byte[].
 */
public class MqttMessage {

	private byte[] payload;
	private int qos = MqttClient.MQTT_QOS1;
	private boolean retained = false;

	/**
	 * Utility method to validate the supplied QoS value.
	 *
	 * @param qos
	 *            The QoS Level
	 * @throws IllegalArgumentException
	 *             if value of QoS is not 0, 1 or 2.
	 */
	public static void validateQos(int qos) {
		if ((qos < MqttClient.MQTT_QOS0) || (qos > MqttClient.MQTT_QOS1)) {
			throw new IllegalArgumentException();
		}
	}

	/**
	 * Constructs a message with an empty payload, and all other values set to defaults.
	 *
	 * The defaults are:
	 * <ul>
	 * <li>Message QoS set to 1</li>
	 * <li>Message will not be "retained" by the server</li>
	 * </ul>
	 */
	public MqttMessage() {
		this(new byte[] {});
	}

	/**
	 * Constructs a message with the specified byte array as a payload, and all other values set to defaults.
	 *
	 * @param payload
	 *            The Bytearray of the payload. <b>Important Note</b> : The array is not cloned for memory efficiency
	 *            reasons. Be careful with it.
	 */
	public MqttMessage(byte[] payload) {
		this.payload = payload; // NOSONAR The array is not cloned for memory efficiency reasons.
		// Be careful with it.
	}

	/**
	 * Returns the payload as a byte array.
	 *
	 * @return the payload as a byte array. <b>Important Note</b> : The array is not cloned for memory efficiency
	 *         reasons. Be careful with it.
	 */
	public byte[] getPayload() {
		return this.payload; // NOSONAR The array is not cloned for memory efficiency reasons.
		// Be careful with it.
	}

	/**
	 * Sets the payload of this message to be the specified byte array.
	 *
	 * @param payload
	 *            the payload for this message. <b>Important Note</b> : The array is not cloned for memory efficiency
	 *            reasons. Be careful with it.
	 */
	public void setPayload(byte[] payload) {
		this.payload = payload; // NOSONAR The array is not cloned for memory efficiency reasons.
		// Be careful with it.
	}

	/**
	 * Returns whether or not this message should be/was retained by the server. For messages received from the server,
	 * this method returns whether or not the message was from a current publisher, or was "retained" by the server as
	 * the last message published on the topic.
	 *
	 * @return <code>true</code> if the message should be, or was, retained by the server.
	 * @see #setRetained(boolean)
	 */
	public boolean isRetained() {
		return this.retained;
	}

	/**
	 * Whether or not the publish message should be retained by the messaging engine. Sending a message with retained
	 * set to <code>true</code> and with an empty byte array as the payload e.g. <code>new byte[0]</code> will clear the
	 * retained message from the server. The default value is <code>false</code>
	 *
	 * @param retained
	 *            whether or not the messaging engine should retain the message.
	 */
	public void setRetained(boolean retained) {
		this.retained = retained;
	}

	/**
	 * Returns the quality of service for this message.
	 *
	 * @return the quality of service to use, either 0, 1.
	 * @see #setQos(int)
	 */
	public int getQos() {
		return this.qos;
	}

	/**
	 * Sets the quality of service for this message.
	 * <ul>
	 * <li>Quality of Service 0 - indicates that a message should be delivered at most once (zero or one times). The
	 * message will not be persisted to disk, and will not be acknowledged across the network. This QoS is the fastest,
	 * but should only be used for messages which are not valuable. Also known as "fire and forget".</li>
	 *
	 * <li>Quality of Service 1 - indicates that a message should be delivered at least once (one or more times). The
	 * message will be acknowledged across the network. This is the default QoS.</li>
	 *
	 * </ul>
	 *
	 * @param qos
	 *            the "quality of service" to use. Set to 0 or 1.
	 * @throws IllegalArgumentException
	 *             if value of QoS is not 0 or 1.
	 * @throws IllegalStateException
	 *             if this message cannot be edited
	 */
	public void setQos(int qos) {
		validateQos(qos);
		this.qos = qos;
	}

	/**
	 * Returns a string representation of this message's payload. Makes an attempt to return the payload as a string. As
	 * the MQTT client has no control over the content of the payload it may fail.
	 *
	 * @return a string representation of this message.
	 */
	@Override
	public String toString() {
		return new String(this.payload);
	}

}
