/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util;

import ej.bluetooth.BluetoothDescriptor;
import ej.bluetooth.BluetoothPermissions;
import ej.bluetooth.BluetoothUuid;

/**
 * Service bluetooth util.
 */
public class BluetoothServicesUtil {

	/**
	 * User description.
	 */
	public static final BluetoothUuid CUD_UUID = new BluetoothUuid(0x2901);

	/**
	 * Client characteristic configuration.
	 */
	public static final BluetoothUuid CCC_UUID = new BluetoothUuid(0x2902);

	/**
	 * Enable notifications.
	 */
	private static final byte NOTIFICATIONS_ENABLED = 0x01;
	/**
	 * Enable indications.
	 */
	private static final byte INDICATIONS_ENABLED = 0x02;

	private BluetoothServicesUtil() {
		// private constructor
	}

	/**
	 * Creates a new CUD descriptor.
	 *
	 * @return a new CUD descriptor.
	 */
	public static BluetoothDescriptor createCUD() {
		return new BluetoothDescriptor(CUD_UUID, BluetoothPermissions.READ);
	}

	/**
	 * Creates a new Client characteristic configuration descriptor.
	 *
	 * @return a new CCC descriptor.
	 */
	public static BluetoothDescriptor createCCC() {
		return new BluetoothDescriptor(CCC_UUID, BluetoothPermissions.RW);
	}

	/**
	 * Checks if a CCC value has notification enabled.
	 *
	 * @param value
	 *            the CCC value to check.
	 * @return <code>true</code> if the notification is enabled.
	 */
	public static boolean checkCccNotifications(byte[] value) {
		if (value.length != 2) {
			throw new IllegalArgumentException();
		}

		return (value[0] & NOTIFICATIONS_ENABLED) != 0;
	}

	/**
	 * Checks if a CCC value has indication enabled.
	 *
	 * @param value
	 *            the CCC value to check.
	 * @return <code>true</code> if the indication is enabled.
	 */
	public static boolean checkCccNIndication(byte[] value) {
		if (value.length != 2) {
			throw new IllegalArgumentException();
		}

		return (value[0] & INDICATIONS_ENABLED) != 0;
	}

	/**
	 * Makes a CCC value.
	 *
	 * @param notifications
	 *            <code>true</code> if the notification is enabled.
	 * @param indications
	 *            <code>true</code> if the indications is enabled.
	 * @return a new CCC value with the given characteristics.
	 */
	public static byte[] makeCccValue(boolean notifications, boolean indications) {
		byte flags = 0;
		if (notifications) {
			flags |= NOTIFICATIONS_ENABLED;
		} else if (indications) {
			flags |= INDICATIONS_ENABLED;
		}

		return new byte[] { flags, 0 };
	}
}
