/*
 * Copyright 2019-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.storage.util;

import ej.util.message.Level;

/**
 * Contains tools related to storage.
 */
public class StorageHelper {

	private StorageHelper() {
	}

	/**
	 * Checks that a storage ID is valid.
	 *
	 * @param id
	 *            the storage ID to check.
	 * @throws IllegalArgumentException
	 *             if the given storage identifier is not a valid identifier.
	 */
	public static void checkID(String id) {
		if (id.length() == 0) {
			throw new IllegalArgumentException(StorageMessage.BUILDER.buildMessage(Level.SEVERE,
					StorageMessage.CATEGORY, StorageMessage.ID_IS_ZERO_LENGTH));
		}
		if (!isLetter(id.charAt(0))) {
			throw new IllegalArgumentException(StorageMessage.BUILDER.buildMessage(Level.SEVERE,
					StorageMessage.CATEGORY, StorageMessage.ID_DOES_NOT_START_WITH_LETTER));
		}

		for (int i = 1; i < id.length(); i++) {
			if (!isValidPart(id.charAt(i))) {
				throw new IllegalArgumentException(
						StorageMessage.BUILDER.buildMessage(Level.SEVERE, StorageMessage.CATEGORY,
								StorageMessage.ID_CONTAINS_INVALID_CHARACTER, Character.toString(id.charAt(i))));
			}
		}
	}

	/**
	 * Checks whether a storage ID is valid or not.
	 *
	 * @param id
	 *            the storage ID to check.
	 * @return <code>true</code> if the ID is valid, <code>false</code> otherwise.
	 */
	public static boolean isValidID(String id) {
		if (id.length() == 0 || !isLetter(id.charAt(0))) {
			return false;
		}

		for (int i = 1; i < id.length(); i++) {
			if (!isValidPart(id.charAt(i))) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Checks that an offset is valid.
	 *
	 * @param offset
	 *            the offset to check.
	 * @throws IllegalArgumentException
	 *             if the given offset is negative.
	 */
	public static void checkOffset(long offset) {
		if (offset < 0) {
			throw new IllegalArgumentException(StorageMessage.BUILDER.buildMessage(Level.SEVERE,
					StorageMessage.CATEGORY, StorageMessage.OFFSET_IS_NEGATIVE));
		}
	}

	private static boolean isLetter(char c) {
		return Character.isLowerCase(c) || Character.isUpperCase(c) || '_' == c;
	}

	private static boolean isValidPart(char c) {
		return isLetter(c) || Character.isDigit(c) || '-' == c || '.' == c;
	}

}
