/*
 * Java
 *
 * Copyright 2012-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.benchmark;

import ej.benchmark.time.PlatformTimeStrategy;
import ej.benchmark.time.TimeStrategy;

/**
 * This helper class proposes a method to run a bench respecting the
 * specification described in {@link BenchTest}.
 */
public class BenchHelper {

	private BenchHelper() {
		// hidden constructor (static methods only)
	}

	/**
	 * Identical to {@link #run(BenchTest, TimeStrategy)} passing
	 * {@link PlatformTimeStrategy} as {@link TimeStrategy}.
	 *
	 * @param test the test to bench
	 * @return the bench result
	 */
	public static BenchResult run(BenchTest test) {
		return run(test, new PlatformTimeStrategy());
	}

	/**
	 * Runs the test several time and returns the time spent.
	 *
	 * @param test the test to run
	 * @param time the time strategy
	 * @return the bench result
	 */
	public static BenchResult run(BenchTest test, TimeStrategy time) {

		// 1) prepare test environnement
		test.pre();

		// 2) retrieve the number of iterations to perform
		int iterations = test.getNumberOfIterations();

		// 3) run empty test
		long startEmptyTime = time.getTime();
		for (int i = 0; i < iterations; ++i) {
			test.emptyRun();
		}
		long emptyRunTime = time.getTime() - startEmptyTime;

		// 4) run full test
		long startTime = time.getTime();
		for (int i = 0; i < iterations; ++i) {
			test.run();
		}
		long fullTime = time.getTime() - startTime;

		// 5) clean test environment
		test.post();

		// sanity check
		if (emptyRunTime > fullTime) {
			throw new AssertionError("An error occured, empty run was shorter than normal run"); //$NON-NLS-1$
		}

		return new BenchResult(test, iterations, emptyRunTime, fullTime);
	}
}
