/*
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.outline.border;

import ej.annotation.Nullable;
import ej.microui.display.GraphicsContext;
import ej.mwt.style.outline.FlexibleOutline;
import ej.mwt.util.Size;

/**
 * Represents and draw a rectangular colored border with one color.
 * <p>
 * The thicknesses are stored as a <code>char</code> for heap optimization and therefore cannot exceed
 * <code>65535</code>.
 */
public class FlexibleRectangularBorder extends FlexibleOutline {

	private final int color;

	/**
	 * Creates a flexible rectangular border specifying its color and edges thickness.
	 * <p>
	 * The given thickness values are clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param color
	 *            the color.
	 * @param top
	 *            the top thickness.
	 * @param right
	 *            the right thickness.
	 * @param bottom
	 *            the bottom thickness.
	 * @param left
	 *            the left thickness.
	 */
	public FlexibleRectangularBorder(int color, int top, int right, int bottom, int left) {
		super(top, right, bottom, left);
		this.color = color;
	}

	/**
	 * Gets the color.
	 *
	 * @return the color.
	 */
	public int getColor() {
		return this.color;
	}

	@Override
	public void apply(GraphicsContext g, Size size) {
		int width = size.getWidth();
		int height = size.getHeight();

		int top = getTop();
		int bottom = getBottom();
		int left = getLeft();
		int right = getRight();

		g.setColor(this.color);
		BorderHelper.drawRectangularBorder(g, width, height, left, top, right, bottom);

		super.apply(g, size);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && getClass() == obj.getClass()) {
			FlexibleRectangularBorder border = (FlexibleRectangularBorder) obj;
			return this.color == border.color && super.equals(obj);
		}
		return false;
	}

	@Override
	public int hashCode() {
		return 15 * this.color + super.hashCode();
	}

}
