/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.background;

import ej.annotation.Nullable;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Image;
import ej.microui.display.Painter;
import ej.mwt.util.Alignment;

/**
 * Draws an image on the background.
 */
public class ImageBackground implements Background {

	private final Image image;
	private final byte horizontalAlignment;
	private final byte verticalAlignment;

	/**
	 * Creates an image background aligned on the top-left corner.
	 *
	 * @param image
	 *            the background image to set.
	 */
	public ImageBackground(Image image) {
		this(image, Alignment.LEFT, Alignment.TOP);
	}

	/**
	 * Creates an image background specifying its alignment.
	 *
	 * @param image
	 *            the background image to set.
	 * @param horizontalAlignment
	 *            the horizontal alignment to set.
	 * @param verticalAlignment
	 *            the vertical alignment to set.
	 * @throws IllegalArgumentException
	 *             if the horizontal or vertical alignment is not valid.
	 * @see Alignment#validateHorizontalAlignment(int)
	 * @see Alignment#validateVerticalAlignment(int)
	 */
	public ImageBackground(Image image, int horizontalAlignment, int verticalAlignment) {
		Alignment.validateHorizontalAlignment(horizontalAlignment);
		Alignment.validateVerticalAlignment(verticalAlignment);
		this.image = image;
		this.horizontalAlignment = (byte) horizontalAlignment;
		this.verticalAlignment = (byte) verticalAlignment;
	}

	@Override
	public boolean isTransparent(int width, int height) {
		Image image = this.image;
		return (image.isTransparent() || image.getWidth() < width || image.getHeight() < height);
	}

	@Override
	public void apply(GraphicsContext g, int width, int height) {
		// draw image
		Image image = this.image;
		int x = Alignment.computeLeftX(image.getWidth(), 0, width, this.horizontalAlignment);
		int y = Alignment.computeTopY(image.getHeight(), 0, height, this.verticalAlignment);
		Painter.drawImage(g, image, x, y);

		// remove background color
		g.removeBackgroundColor();
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj instanceof ImageBackground) {
			ImageBackground background = (ImageBackground) obj;
			return this.image == background.image && this.horizontalAlignment == background.horizontalAlignment
					&& this.verticalAlignment == background.verticalAlignment;
		}
		return false;
	}

	@Override
	public int hashCode() {
		// don't call this.image.hashCode() to avoid embedding the hashCode() method of every class (M0081MEJA-1240)
		return this.horizontalAlignment * this.verticalAlignment;
	}
}
