/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.util;

import ej.microui.display.GraphicsContext;
import ej.mwt.style.Style;

/**
 * Provides utility methods to apply outlines and backgrounds.
 */
public class OutlineHelper {

	// Prevents initialization.
	private OutlineHelper() {
	}

	/**
	 * Applies all the outlines of the given style on the given object.
	 * <p>
	 * The given object must include the outlines of the widget. It is modified to the content bounds/size of the
	 * widget.
	 *
	 * @param outlineable
	 *            the outlineable.
	 * @param style
	 *            the style to apply.
	 * @see ej.mwt.style.outline.Outline#apply(Outlineable)
	 */
	public static void applyOutlines(Outlineable outlineable, Style style) {
		style.getMargin().apply(outlineable);
		style.getBorder().apply(outlineable);
		style.getPadding().apply(outlineable);
	}

	/**
	 * Applies the different outlines and the background on the given graphics context and the given size.
	 * <p>
	 * The given size must include the outlines of the widget. It is modified to the content size of the widget.
	 * <p>
	 * The application order is: margin, background, border, padding.
	 *
	 * @param g
	 *            the graphics context to modify.
	 * @param size
	 *            the size.
	 * @param style
	 *            the style to apply.
	 * @see ej.mwt.style.outline.Outline#apply(GraphicsContext, Size)
	 * @see ej.mwt.style.background.Background#apply(GraphicsContext, int, int)
	 */
	public static void applyOutlinesAndBackground(GraphicsContext g, Size size, Style style) {
		style.getMargin().apply(g, size);
		style.getBackground().apply(g, size.getWidth(), size.getHeight());
		style.getBorder().apply(g, size);
		style.getPadding().apply(g, size);
	}
}
