/*
 * Java
 *
 * Copyright 2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.event;

/**
 * EventQueueListener is an interface to be implemented by parties interested to receive events of a specific type from
 * an {@link EventQueue}.
 * <p>
 * An event queue listener can be registered to an event queue using
 * {@link EventQueue#registerListener(EventQueueListener, int)}.
 * <p>
 * Before registering your listener, you must get a valid unique type using the {@link EventQueue#getNewType()} method
 * from the {@link EventQueue} class. Then you can register your listener using
 * {@link EventQueue#registerListener(EventQueueListener, int)}.
 */
public interface EventQueueListener {

	/**
	 * Handles an event.
	 * <p>
	 * This method is called by the queue when an event with the right type is read.
	 * <p>
	 *
	 * @param type
	 *            the type of the event to handle.
	 * @param data
	 *            the data of the event to handle.
	 */
	void handleEvent(int type, int data);

	/**
	 * Handles an extended event.
	 * <p>
	 * This method is called by the queue when an extended event with the right type is read.
	 * <p>
	 *
	 * @param type
	 *            the type of the event to handle.
	 * @param eventDataReader
	 *            the EventDataReader of the event to handle.
	 */
	void handleExtendedEvent(int type, EventDataReader eventDataReader);

}
