/*
 * Java
 *
 * Copyright 2016-2020 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util;

import org.eclipse.core.runtime.IProgressMonitor;

import com.microej.kf.util.message.KfUtilMessage;

import ej.kf.Kernel;
import ej.kf.Module;
import ej.kf.Proxy;
import ej.util.message.Level;

/**
 *
 */
public class IProgressMonitorProxy extends Proxy<IProgressMonitor> implements IProgressMonitor {

	private final Proxy<IProgressMonitor> proxy;

	public IProgressMonitorProxy(Proxy<IProgressMonitor> proxy) {
		this.proxy = proxy;
	}

	@Override
	public void beginTask(String name, int totalWork) {
		Kernel.enter();
		final IProgressMonitor target = this.proxy.getReference();
		if(target == null){
			return;
		}
		Module owner = Kernel.getOwner(target);

		try {
			target.beginTask(Kernel.clone(name, owner), totalWork);
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.PROGRESS_MONITOR_ERROR, t);
		}
	}

	@Override
	public void done() {
		Kernel.enter();
		try {
			this.proxy.getReference().done();
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.PROGRESS_MONITOR_ERROR, t);
		}
	}

	@Override
	public void internalWorked(double work) {
		Kernel.enter();
		try {
			this.proxy.getReference().internalWorked(work);
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.PROGRESS_MONITOR_ERROR, t);
		}
	}

	@Override
	public boolean isCanceled() {
		Kernel.enter();
		try {
			return this.proxy.getReference().isCanceled();
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.PROGRESS_MONITOR_ERROR, t);
			return false;
		}
	}

	@Override
	public void setCanceled(boolean value) {
		Kernel.enter();
		try {
			this.proxy.getReference().setCanceled(value);
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.PROGRESS_MONITOR_ERROR, t);
		}
	}

	@Override
	public void setTaskName(String name) {
		Kernel.enter();
		final IProgressMonitor target = this.proxy.getReference();
		if(target == null){
			return;
		}
		Module owner = Kernel.getOwner(target);

		try {
			target.setTaskName(Kernel.clone(name, owner));
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.PROGRESS_MONITOR_ERROR, t);
		}
	}

	@Override
	public void subTask(String name) {
		Kernel.enter();
		final IProgressMonitor target = this.proxy.getReference();
		if(target == null){
			return;
		}
		Module owner = Kernel.getOwner(target);

		try {
			target.subTask(Kernel.clone(name, owner));
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.PROGRESS_MONITOR_ERROR, t);
		}
	}

	@Override
	public void worked(int work) {
		Kernel.enter();
		try {
			this.proxy.getReference().worked(work);
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.PROGRESS_MONITOR_ERROR, t);
		}
	}
}
