/*
 * Copyright (c) 2022, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2022 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

package com.microej.stringregex;

import java.util.regex.PatternSyntaxException;

import org.apache.regexp.RE;
import org.apache.regexp.RESyntaxException;

import ej.basictool.ArrayTools;
import ej.basictool.annotation.Extend;

/**
 * Extension regex methods for the {@link String} class.
 */
public class StringRegexExtension {

	/**
	 * Replaces the first substring of this string that matches the given regular expression with the given replacement.
	 *
	 * @param self
	 *            the original string, corresponding to {@code this}, to replace the first match from
	 * @param regex
	 *            the regular expression to which this string is to be matched
	 * @param replacement
	 *            the string to be substituted for the first match
	 *
	 * @return The resulting <tt>String</tt>
	 *
	 * @throws PatternSyntaxException
	 *             if the regular expression's syntax is invalid
	 *
	 */
	@Extend(className = "java.lang.String")
	public static String replaceFirst(String self, String regex, String replacement) {
		String subst;
		if ("".equals(regex)) {
			subst = replacement + self;
		} else {
			try {
				RE re = new RE(regex);
				subst = re.subst(self, replacement, RE.REPLACE_FIRSTONLY + RE.REPLACE_BACKREFERENCES);
			} catch (RESyntaxException rese) {
				throw new PatternSyntaxException(rese);
			}
		}
		return subst;
	}

	/**
	 * Replaces each substring of this string that matches the given regular expression with the given replacement.
	 *
	 * @param self
	 *            the original string, corresponding to {@code this}, to replace matches from
	 * @param regex
	 *            the regular expression to which this string is to be matched
	 * @param replacement
	 *            the string to be substituted for each match
	 *
	 * @return The resulting <tt>String</tt>
	 *
	 * @throws PatternSyntaxException
	 *             if the regular expression's syntax is invalid
	 */
	@Extend(className = "java.lang.String")
	public static String replaceAll(String self, String regex, String replacement) {
		String subst;
		if ("".equals(regex)) {
			StringBuilder stringBuilder = new StringBuilder(replacement);
			for (char c : self.toCharArray()) {
				stringBuilder.append(c);
				stringBuilder.append(replacement);
			}
			subst = stringBuilder.toString();
		} else {
			try {
				RE re = new RE(regex);
				subst = re.subst(self, replacement, RE.REPLACE_ALL + RE.REPLACE_BACKREFERENCES);
			} catch (RESyntaxException rese) {
				throw new PatternSyntaxException(rese);
			}
		}
		return subst;
	}

	/**
	 * Splits this string around matches of the given regular expression.
	 *
	 * Trailing empty strings are not included in the resulting array.
	 *
	 * @param self
	 *            the original string, corresponding to {@code this}, to be split
	 * @param regex
	 *            the delimiting regular expression
	 *
	 * @return the array of strings computed by splitting this string around matches of the given regular expression
	 *
	 * @throws PatternSyntaxException
	 *             if the regular expression's syntax is invalid
	 */
	@Extend(className = "java.lang.String")
	public static String[] split(String self, String regex) {
		String[] fullSplit;
		try {
			RE re = new RE(regex);
			fullSplit = re.split(self);
		} catch (RESyntaxException rese) {
			throw new PatternSyntaxException(rese);
		}
		// re.split(String) leaves trailing empty strings in the array, but the original
		// Pattern.split(String) that we want to comply to is equivalent to having a
		// limit argument set to 0, which removes trailing empty strings. So the
		// following loop counts the number of them, if any, to shrink off the array at return.
		int nbTrailingEmpties = 0;
		for (int i = fullSplit.length - 1; i >= 0; i--) {
			if (fullSplit[i].equals("")) {
				nbTrailingEmpties++;
			} else {
				break;
			}
		}
		return ArrayTools.shrink(fullSplit, fullSplit.length - nbTrailingEmpties, nbTrailingEmpties);
	}

	/**
	 * Tells whether or not this string matches the given regular expression.
	 *
	 * @param self
	 *            the original string, corresponding to {@code this}, to check for matching
	 * @param regex
	 *            the regular expression to which this string is to be matched
	 *
	 * @return <tt>true</tt> if, and only if, this string matches the given regular expression
	 *
	 * @throws PatternSyntaxException
	 *             if the regular expression's syntax is invalid
	 */
	@Extend(className = "java.lang.String")
	public static boolean matches(String self, String regex) {
		boolean match;
		try {
			/*
			 * re.match(string) returns true if one match is found in the string, not only if the whole string matches.
			 * This behaviour is more similar to Matcher.find() rather than Matcher.matches(). To work around this, we
			 * wrap the regex between "^(?:" and ")$" to ensure it matches a full string.
			 */
			if (regex != null) {
				regex = "^(?:" + regex + ")$";
			}
			RE re = new RE(regex);
			match = re.match(self);
		} catch (RESyntaxException rese) {
			throw new PatternSyntaxException(rese);
		}
		return match;
	}
}
