/*
 * Java
 *
 * Copyright 2018-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.net.util;

import java.io.IOException;

import android.net.SntpClient;
import ej.bon.Util;
import ej.util.message.Level;

/**
 * Utilitary functions to update the local time of the board..
 */
public class NtpUtil {

	/**
	 * Property to set the NTP URL.
	 */
	public static final String NTP_URL_PROPERTY = "ntp.url"; //$NON-NLS-1$

	/**
	 * Property to set the NTP port.
	 */
	public static final String NTP_PORT_PROPERTY = "ntp.port"; //$NON-NLS-1$

	/**
	 * Property to set the NTP timeout.
	 */
	public static final String NTP_TIMEOUT_PROPERTY = "ntp.timeout"; //$NON-NLS-1$

	/**
	 * Property to set the NTP attempts.
	 */
	public static final String NTP_ATTEMPTS_PROPERTY = "ntp.attempts"; //$NON-NLS-1$

	/**
	 * Default ntp port.
	 */
	public static final int NTP_DEFAULT_PORT = 123;

	/**
	 * Default NTP url.
	 */
	public static final String NTP_DEFAULT_SERVER = "0.pool.ntp.org"; //$NON-NLS-1$

	/**
	 * Default NTP timeout.
	 */
	public static final int NTP_DEFAULT_TIMEOUT = 1000;

	/**
	 * Default NTP attempts.
	 */
	public static final int NTP_DEFAULT_ATTEMPTS = 3;

	private NtpUtil() {
		// Forbid instantiation.
	}

	/**
	 * Updates the local time. It uses the properties {@value #NTP_URL_PROPERTY}, {@value #NTP_PORT_PROPERTY} and
	 * {@value #NTP_TIMEOUT_PROPERTY} to gets the server.
	 *
	 * The default values are {@value #NTP_DEFAULT_SERVER}:{@value #NTP_DEFAULT_PORT} with a timeout
	 * {@value #NTP_DEFAULT_TIMEOUT}.
	 *
	 * @throws IOException
	 *             if an error occurred.
	 */
	public static void updateLocalTime() throws IOException {
		String url = System.getProperty(NTP_URL_PROPERTY, NTP_DEFAULT_SERVER);
		int port = Integer.getInteger(NTP_PORT_PROPERTY, NTP_DEFAULT_PORT).intValue();
		int timeout = Integer.getInteger(NTP_TIMEOUT_PROPERTY, NTP_DEFAULT_TIMEOUT).intValue();
		updateLocalTime(url, port, timeout);
	}

	/**
	 * Updates the local time.
	 *
	 * @param url
	 *            the ntp server url.
	 * @param port
	 *            the ntp server port.
	 * @param timeout
	 *            the timeout.
	 * @throws IOException
	 *             if an error occurred.
	 */
	public static void updateLocalTime(String url, int port, int timeout) throws IOException {
		int attempts = Integer.getInteger(NTP_ATTEMPTS_PROPERTY, NTP_DEFAULT_ATTEMPTS).intValue();

		SntpClient ntpClient = new SntpClient();

		for (int i = 0; i < attempts; i++) {
			if (ntpClient.requestTime(url, port, timeout)) {
				long now = ntpClient.getNtpTime() + Util.platformTimeMillis() - ntpClient.getNtpTimeReference();
				Util.setCurrentTimeMillis(now);
				return;
			}
		}

		throw new IOException(
				Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.ERROR_NTP_REQUEST));
	}
}
