/*
 * Copyright 2016-2019 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.property.loader;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.property.PropertyLoader;
import ej.property.PropertyPermission;

/**
 * A property loader that searches in system properties.
 *
 * @see System#getProperty(String)
 * @see System#getProperty(String, String)
 */
public class SystemPropertyLoader implements PropertyLoader {

	@Override
	@Nullable
	public String getProperty(String key) {
		checkSecurity(key);
		return System.getProperty(key);
	}

	@Override
	public String getProperty(String key, String defaultValue) {
		checkSecurity(key);
		return System.getProperty(key, defaultValue);
	}

	private void checkSecurity(String key) {
		SecurityManager securityManager = System.getSecurityManager();
		if (securityManager != null) {
			@SuppressWarnings("null") // Class name is not null for sure.
			@NonNull
			String className = getClass().getSimpleName();
			securityManager.checkPermission(new PropertyPermission(className, key, PropertyPermission.GET_ACTION));
		}
	}

}
