/*
 * Copyright 2011-2018 IS2T. All rights reserved.
 *
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.basictool;

/**
 * Helper to manage bit fields.
 */
public class BitFieldHelper {

	private BitFieldHelper() {
	}

	/**
	 * Sets a boolean value in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param value
	 *            the value to set.
	 * @param mask
	 *            the bit mask of the value in the int.
	 * @return the modified bit field.
	 */
	public static int setBooleanProperty(int field, boolean value, int mask) {
		if (value) {
			field |= mask;
		} else {
			field &= ~mask;
		}
		return field;
	}

	/**
	 * Forces the boolean to true in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param value
	 *            the value to set.
	 * @param offset
	 *            the offset of the boolean in the int.
	 * @return the modified bit field.
	 */
	public static int setBooleanPropertyAtOffset(int field, boolean value, int offset) {
		if (value) {
			return setBooleanPropertyAtOffset(field, offset);
		} else {
			return unsetBooleanPropertyAtOffset(field, offset);
		}
	}

	/**
	 * Forces the boolean to true in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param mask
	 *            the bit mask of the value in the int.
	 * @return the modified bit field.
	 */
	public static int setBooleanProperty(int field, int mask) {
		return field |= mask;
	}

	/**
	 * Forces the boolean to true in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param offset
	 *            the offset of the boolean in the int.
	 * @return the modified bit field.
	 */
	public static int setBooleanPropertyAtOffset(int field, int offset) {
		return field |= (0x1 << offset);
	}

	/**
	 * Forces the boolean to false in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param mask
	 *            the bit mask of the value in the int.
	 * @return the modified bit field.
	 */
	public static int unsetBooleanProperty(int field, int mask) {
		return field &= ~mask;
	}

	/**
	 * Forces the boolean to false in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param offset
	 *            the offset of the boolean in the int.
	 * @return the modified bit field.
	 */
	public static int unsetBooleanPropertyAtOffset(int field, int offset) {
		return field &= ~(0x1 << offset);
	}

	/**
	 * Gets a boolean value in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param mask
	 *            the bit mask of the value in the int.
	 * @return the boolean value.
	 */
	public static boolean getBooleanProperty(int field, int mask) {
		return (field & mask) != 0x0;
	}

	/**
	 * Gets a boolean value in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param offset
	 *            the offset of the boolean in the int.
	 * @return the boolean value.
	 */
	public static boolean getBooleanPropertyAtOffset(int field, int offset) {
		return ((field >>> offset) & 0x1) != 0x0;
	}

	/**
	 * Sets a int value in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param value
	 *            the value to set.
	 * @param mask
	 *            the bit mask of the value in the int.
	 * @param offset
	 *            the offset of the value in the int.
	 * @return the modified bit field.
	 */
	public static int setIntProperty(int field, int value, int mask, int offset) {
		field &= ~mask; // erase previous value
		field |= (value << offset) & mask;
		return field;
	}

	/**
	 * Gets a int value in a bit field.
	 *
	 * @param field
	 *            the bit field.
	 * @param mask
	 *            the bit mask of the value in the int.
	 * @param offset
	 *            the offset of the value in the int.
	 * @return the int value.
	 */
	public static int getIntProperty(int field, int mask, int offset) {
		return (field & mask) >>> offset;
	}

}
