/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.widget.basic;

import ej.annotation.Nullable;
import ej.microui.event.Event;
import ej.microui.event.generator.Pointer;
import ej.mwt.event.DesktopEventGenerator;
import ej.mwt.event.PointerEventDispatcher;

/**
 * A button is a widget that displays a text and reacts to click events.
 * <p>
 * This example shows a simple button:
 *
 * <pre>
 * Button button = new Button("Press Me");
 * </pre>
 *
 * <img src="doc-files/button-simple.png" alt="Simple button.">
 * <p>
 * This example shows a styled button:
 *
 * <pre>
 * Button button = new Button("Press Me");
 *
 * CascadingStylesheet stylesheet = new CascadingStylesheet();
 * desktop.setStylesheet(stylesheet);
 *
 * EditableStyle buttonStyle = stylesheet.getSelectorStyle(new TypeSelector(Button.class));
 * buttonStyle.setBackground(new RectangularBackground(Colors.SILVER));
 * buttonStyle.setBorder(new RectangularBorder(Colors.BLACK, 1));
 * buttonStyle.setFont(Font.getFont("/fonts/source_sans_pro_24.ejf"));
 * </pre>
 *
 * <img src="doc-files/button-styled.png" alt="Styled button.">
 * <p>
 * This example shows a styled button with its background color changing on a press action:
 *
 * <pre>
 * Button button = new Button("Press Me");
 *
 * CascadingStylesheet stylesheet = new CascadingStylesheet();
 * desktop.setStylesheet(stylesheet);
 *
 * EditableStyle buttonStyle = stylesheet.getSelectorStyle(new TypeSelector(Button.class));
 * buttonStyle.setBackground(new RectangularBackground(Colors.SILVER));
 * buttonStyle.setBorder(new RectangularBorder(Colors.BLACK, 1));
 * buttonStyle.setFont(Font.getFont("/fonts/source_sans_pro_24.ejf"));
 *
 * EditableStyle pressedButtonStyle = stylesheet
 * 		.getSelectorStyle(new AndCombinator(new TypeSelector(Button.class), new StateSelector(States.ACTIVE)));
 * pressedButtonStyle.setBackground(new RectangularBackground(Colors.GRAY));
 * </pre>
 *
 * <img src="doc-files/button-styled-pressed.png" alt="Styled pressed button.">
 *
 * @since 2.3.0
 */
public class Button extends Label {

	/**
	 * Active state.
	 */
	public static final int ACTIVE = 1;

	private @Nullable OnClickListener onClickListener;
	private boolean pressed;

	/**
	 * Creates a button with an empty text.
	 */
	public Button() {
		this("");
	}

	/**
	 * Creates a button with the given text to display.
	 *
	 * @param text
	 *            the text to display.
	 */
	public Button(String text) {
		super(text);

		this.onClickListener = null;
		this.pressed = false;
	}

	@Override
	protected void onAttached() {
		setEnabled(true);
		super.onAttached();
	}

	/**
	 * Sets the listener on the click events of this button.
	 *
	 * @param listener
	 *            the listener to set.
	 */
	public void setOnClickListener(@Nullable OnClickListener listener) {
		this.onClickListener = listener;
	}

	@Override
	public boolean isInState(int state) {
		return (state == ACTIVE && this.pressed) || super.isInState(state);
	}

	@Override
	public boolean handleEvent(int event) {
		int type = Event.getType(event);
		if (type == Pointer.EVENT_TYPE) {
			int action = Pointer.getAction(event);
			if (action == Pointer.PRESSED) {
				setPressed(true);
			} else if (action == Pointer.RELEASED) {
				setPressed(false);
				handleClick();
				return true;
			}
		} else if (type == DesktopEventGenerator.EVENT_TYPE) {
			int action = DesktopEventGenerator.getAction(event);
			if (action == PointerEventDispatcher.EXITED) {
				setPressed(false);
			}
		}

		return super.handleEvent(event);
	}

	private void setPressed(boolean pressed) {
		this.pressed = pressed;
		updateStyle();
		requestRender();
	}

	/**
	 * Handles a click event.
	 */
	public void handleClick() {
		OnClickListener listener = this.onClickListener;
		if (listener != null) {
			listener.onClick();
		}
	}
}
