/*
 * Java
 *
 * Copyright 2018-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util.services.cts;

import java.util.Calendar;

import ej.bluetooth.BluetoothAttribute;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothConnection;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.BluetoothStatus;
import ej.bluetooth.listeners.impl.DefaultLocalServiceListener;
import ej.bluetooth.util.AttributeNotFoundException;
import ej.bluetooth.util.ServiceHelper;
import ej.bon.ByteArray;

/**
 * The <code>CurrentTimeServer</code> class represents a current time server.
 */
public class CurrentTimeServer extends DefaultLocalServiceListener {

	private final BluetoothService service;
	private final BluetoothCharacteristic currentTimeChar;
	private final BluetoothCharacteristic localTimeInfoChar;

	/**
	 * Creates a current time server, using the current time service provided by this device.
	 *
	 * @param service
	 *            the current time service provided by this device.
	 * @throws IllegalArgumentException
	 *             if one of the mandatory attributes of the service is missing.
	 */
	public CurrentTimeServer(BluetoothService service) {
		this.service = service;

		try {
			this.currentTimeChar = ServiceHelper.getCharacteristic(service, CurrentTimeConstants.CURRENT_TIME_UUID);
			this.localTimeInfoChar = ServiceHelper.getCharacteristic(service, CurrentTimeConstants.LOCAL_TIME_INFO_UUID);
		} catch (AttributeNotFoundException e) {
			throw new IllegalArgumentException("Invalid current time service");
		}
	}

	/**
	 * Starts this current time server.
	 */
	public void start() {
		this.service.setLocalListener(this);
	}

	/**
	 * Stops this current time server.
	 */
	public void stop() {
		this.service.setLocalListener(new DefaultLocalServiceListener());
	}

	@Override
	public void onReadRequest(BluetoothConnection connection, BluetoothAttribute attribute) {
		if (attribute == this.currentTimeChar) {
			connection.sendReadResponse(attribute, BluetoothStatus.OK, makeCurrentTime());
		} else if (attribute == this.localTimeInfoChar) {
			connection.sendReadResponse(attribute, BluetoothStatus.OK, makeLocalTimeInfo());
		} else {
			super.onReadRequest(connection, attribute);
		}
	}

	private static byte[] makeCurrentTime() {
		Calendar calendar = Calendar.getInstance();

		int localTimeOffset = calendar.get(Calendar.ZONE_OFFSET) + calendar.get(Calendar.DST_OFFSET);
		calendar.set(Calendar.MILLISECOND, calendar.get(Calendar.MILLISECOND) - localTimeOffset);

		int year = calendar.get(Calendar.YEAR);
		int month = calendar.get(Calendar.MONTH) + 1;
		int day = calendar.get(Calendar.DAY_OF_MONTH);
		int hour = calendar.get(Calendar.HOUR_OF_DAY);
		int minute = calendar.get(Calendar.MINUTE);
		int second = calendar.get(Calendar.SECOND);

		int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
		if (dayOfWeek == Calendar.SUNDAY) {
			dayOfWeek = 7;
		} else {
			dayOfWeek = dayOfWeek - 1;
		}

		int fractionsOfSecond = (int) (calendar.getTimeInMillis() * 256 / 1000);
		int adjustReason = 0; // other adjust reasons are not supported

		byte[] value = new byte[10];
		ByteArray.writeShort(value, 0, year, ByteArray.LITTLE_ENDIAN);
		value[2] = (byte) month;
		value[3] = (byte) day;
		value[4] = (byte) hour;
		value[5] = (byte) minute;
		value[6] = (byte) second;
		value[7] = (byte) dayOfWeek;
		value[8] = (byte) fractionsOfSecond;
		value[9] = (byte) adjustReason;
		return value;
	}

	private static byte[] makeLocalTimeInfo() {
		Calendar calendar = Calendar.getInstance();

		int timezoneOffset = calendar.get(Calendar.ZONE_OFFSET) / (15 * 60 * 1000);
		int dstOffset = calendar.get(Calendar.DST_OFFSET) / (15 * 60 * 1000);

		byte[] value = new byte[2];
		value[0] = (byte) timezoneOffset;
		value[1] = (byte) dstOffset;
		return value;
	}
}
