/*
 * Java
 *
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http.encoding;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import ej.hoka.http.HttpConstants;
import ej.hoka.http.HttpRequest;
import ej.hoka.http.HttpResponse;
import ej.hoka.io.IdentityMessageBodyInputStream;
import ej.hoka.io.IdentityMessageBodyOutputStream;

/**
 * Identity transfer coding handler.
 */
public final class IdentityTransferEncoding implements TransferEncoding {

	/**
	 * The private constructor to prevent direct instantiation.
	 */
	private IdentityTransferEncoding() {
		// private constructor, because of singleton behaviour
	}

	/**
	 * Private holder class for the unique instance, lazy initialized when instance is first accessed.
	 */
	private static class IdentityTransferEncodingHolder {
		/**
		 * The static instance to use in factory method.
		 */
		private static final IdentityTransferEncoding INSTANCE = new IdentityTransferEncoding();
	}

	/**
	 * Returns an instance of {@link IdentityTransferEncoding}.
	 *
	 * @return an instance of {@link IdentityTransferEncoding}
	 */
	public static IdentityTransferEncoding getInstance() {
		return IdentityTransferEncodingHolder.INSTANCE;
	}

	/**
	 * Returns an internal ID of this encoding handler.
	 *
	 * @return null
	 */
	@Override
	public String getId() {
		return "identity";
	}

	/**
	 * Creates a new instance of {@link IdentityMessageBodyInputStream} to read the message body of the HTTP request.
	 *
	 * @see IdentityMessageBodyInputStream
	 * @param request
	 *            the HTTP request object.
	 * @param input
	 *            the input stream to read the message body of the HTTP request.
	 * @return {@link IdentityMessageBodyOutputStream}.
	 * @throws IOException
	 *             if an I/O error occurs.
	 */
	@Override
	public InputStream open(HttpRequest request, InputStream input) throws IOException {
		int bodyLength = 0;
		String contentLength = request.getHeader(HttpConstants.HEADER_CONTENT_LENGTH);
		if (contentLength != null) {
			bodyLength = Integer.parseInt(contentLength);
		}
		return new IdentityMessageBodyInputStream(input, bodyLength);
	}

	/**
	 * Creates an {@link IdentityMessageBodyOutputStream} to write the message body of the HTTP response.
	 *
	 * @see IdentityMessageBodyOutputStream
	 * @param response
	 *            the {@link HttpResponse}
	 * @param output
	 *            the {@link OutputStream} to write the message body of the HTTP response
	 * @return {@link IdentityMessageBodyOutputStream}
	 * @throws IOException
	 *             when I/O error occurs
	 */
	@Override
	public OutputStream open(HttpResponse response, OutputStream output) throws IOException {
		return new IdentityMessageBodyOutputStream(output);
	}

}
