/*
 *	Java
 *
 *	Copyright 2015-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import java.lang.ref.WeakReference;

import com.is2t.tools.ArrayTools;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.microui.event.generator.Pointer;

/**
 * Provides link between pointers and widgets
 * <p>
 * Each mouse has a renderable that is under and another that was the one under the mouse when the first button has been
 * pressed. This renderable receives mouse events in the {@link Renderable#handleEvent(int)} method.
 */
/* package */class ExtendedPointer {

	// the extended pointers register
	@NonNull
	private static ExtendedPointer[] Pointers = newExtendedPointers();

	// the peer pointer
	@NonNull
	private final Pointer pointer;
	// the renderable that is under the pointer
	@NonNull
	private WeakReference<Renderable> renderableUnder;
	// the renderable that was under the pointer when pressed
	@NonNull
	private WeakReference<Renderable> renderablePressed;

	/**
	 * Creates a new extended pointer.
	 */
	private ExtendedPointer(@NonNull Pointer pointer) {
		this.pointer = pointer;
		this.renderableUnder = new WeakReference<>(null);
		this.renderablePressed = new WeakReference<>(null);
	}

	/**
	 * @return the renderable under.
	 */
	@Nullable
	/* package */Renderable getRenderableUnder() {
		return this.renderableUnder.get();
	}

	/**
	 * @return the renderable pressed.
	 */
	@Nullable
	/* package */Renderable getRenderablePressed() {
		return this.renderablePressed.get();
	}

	/**
	 * @param renderablePressed
	 *            the renderable to set.
	 */
	/* package */void setRenderablePressed(@Nullable Renderable renderablePressed) {
		this.renderablePressed = new WeakReference<>(renderablePressed);
	}

	/**
	 * @param renderableUnder
	 *            the renderable to set.
	 */
	/* package */void setRenderableUnder(@NonNull Renderable renderableUnder) {
		this.renderableUnder = new WeakReference<>(renderableUnder);
	}

	/**
	 * Gets the extended pointer for the given pointer. If it not yet exists, it is created.
	 */
	/* package */static ExtendedPointer getXPointer(Pointer pointer) {
		ExtendedPointer[] pointers = ExtendedPointer.Pointers;
		for (int i = pointers.length; --i >= 0;) {
			ExtendedPointer xPointer = pointers[i];
			if (xPointer.pointer == pointer) {
				return xPointer;
			}
		}
		// new pointer (not yet registered)
		ExtendedPointer xPointer = new ExtendedPointer(pointer);
		ExtendedPointer.Pointers = (ExtendedPointer[]) ArrayTools.add(pointers, xPointer);
		return xPointer;
	}

	@NonNull
	private static ExtendedPointer[] newExtendedPointers() {
		return new ExtendedPointer[0];
	}
}
