/**
* Java
*
* Copyright 2024 MicroEJ Corp. All rights reserved.
* This library is provided in source code for use, modification and test, subject to license terms.
* Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
*/

package ej.microvg.image;

import java.awt.Color;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import ej.microvg.image.pathdata.PathData;

/**
 * A generator is able to generate images into a GPU format.
 * <p>
 * The implementation of this class is required by the VEE Port. Two implementations are already provided: vglite and
 * nema.
 */
public interface ImageGenerator {

	/**
	 * List of image's formats.
	 */
	public enum Format {
		/** Signed 8-bit format. */
		VG_S8(1),
		/** Signed 16-bit format. */
		VG_S16(2),
		/** Signed 32-bit format. */
		VG_S32(4),
		/** Float 32-bit format. */
		VG_FP32(4);

		private final int unitSize;

		private Format(int unitSize) {
			this.unitSize = unitSize;
		}

		/**
		 * Gets the unit size.
		 *
		 * @return the unit size in number of bytes
		 */
		public int getUnitSize() {
			return this.unitSize;
		}

	}

	/**
	 * List of path commands.
	 */
	public enum Command {
		/** Close command. */
		COMMAND_CLOSE,
		/** Move absolute command. */
		COMMAND_MOVE,
		/** Move relative command. */
		COMMAND_MOVE_REL,
		/** Line absolute command. */
		COMMAND_LINE,
		/** Line relative command. */
		COMMAND_LINE_REL,
		/** Quad absolute command. */
		COMMAND_QUAD,
		/** Quad relative command. */
		COMMAND_QUAD_REL,
		/** Cubic absolute command. */
		COMMAND_CUBIC,
		/** Cubic relative command. */
		COMMAND_CUBIC_REL;
	}

	/**
	 * List of rules to fill a path.
	 */
	public enum FillRule {
		/** Winding fill rule. */
		FILLRULE_NOZERO,
		/** Even-odd fill rule. */
		FILLRULE_EVENODD;

		/**
		 * Converts the fill rule to its encoded byte value.
		 *
		 * @return the encoded value
		 */
		public byte toByte() {
			switch (this) {
			case FILLRULE_NOZERO:
				return 0;
			case FILLRULE_EVENODD:
				return 1;
			default:
				return -1;
			}
		}
	}

	/**
	 * Encodes the GPU command from a {@link Command}
	 *
	 * @param command
	 *            the command to convert.
	 * @return GPU command representation.
	 */
	int encodeCommand(Command command);

	/**
	 * Encodes the GPU color from a {@link Color} and an opacity.
	 *
	 * @param color
	 *            the color to convert.
	 * @param opacity
	 *            the opacity to apply, a value between 0f and 1f.
	 * @return GPU color representation.
	 */
	int encodeColor(Color color, float opacity);

	/**
	 * Encodes a linear gradient for the GPU.
	 *
	 * @param gradient
	 *            the gradient to encode
	 * @return GPU linear gradient representation
	 */
	byte[] encodeGradient(LinearGradient gradient);

	/**
	 * Encodes a path for the GPU (path header + path commands + path commands parameters).
	 * <p>
	 * The {@link Path#encode(PathData)} method should be used to browse the commands of the path. These commands are
	 * encoded by {@link #encodeCommand(Command)} and append to the associated path data.
	 *
	 * @param path
	 *            the path to encode
	 * @return the encoded path
	 */
	byte[] encodePath(Path path);

	/**
	 * Creates a byte buffer for the given size, ordered in little endian.
	 *
	 * @param size
	 *            the byte buffer size
	 * @return the little endian byte buffer
	 */
	default ByteBuffer newByteBuffer(int size) {
		return ByteBuffer.allocate(size).order(ByteOrder.LITTLE_ENDIAN);
	}

}
