/*
 * Java
 *
 * Copyright 2013-2016 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.kf.util.security;

import java.security.Permission;
import java.util.HashMap;

import ej.kf.Feature;
import ej.kf.Kernel;

/**
 * An example of {@link SecurityManager} implementation based on <code>Kernel/Feature</code> runtime model. Implemented
 * checks: <ul><li>A {@link Feature} cannot access a File out of its dedicated space</li></ul>
 */
public class KernelSecurityManager extends SecurityManager {

	/**
	 * Registered permission checks
	 */
	private final HashMap<Class<? extends Permission>, FeaturePermissionCheckDelegate> permissionsMap;

	public KernelSecurityManager() {
		// Initialize permissions checks
		permissionsMap = new HashMap<Class<? extends Permission>, FeaturePermissionCheckDelegate>();
	}

	public void setFeaturePermissionDelegate(Class<? extends Permission> permission,
			FeaturePermissionCheckDelegate delegate) {
		permissionsMap.put(permission, delegate);
	}

	@Override
	public void checkPermission(Permission perm) {
		if (!Kernel.isInKernelMode()) {
			Feature f = (Feature) Kernel.getContextOwner();
			Kernel.enter();
			FeaturePermissionCheckDelegate fp = permissionsMap.get(perm.getClass());
			if (fp != null) {
				fp.checkPermission(perm, f);
			} else {
				unknownPermission(perm);
			}
		}
		// else: Kernel has all the rights: no checks
	}

	/**
	 * This method is called when there is no registered delegate for this kind of Permission. Implementation choice is
	 * to throw a {@link SecurityException} by default.
	 * @param perm
	 * 			the unknown permission
	 */
	protected void unknownPermission(Permission perm) {
		throw new SecurityException();
	}

}
