/*
 * Copyright (c) 2003, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2017-2020 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.util;

import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.Flushable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;

import ej.annotation.Nullable;
import sun.misc.FormattedFloatingDecimal;

/**
 * An interpreter for printf-style format strings. This class provides support for layout justification and alignment,
 * common formats for numeric, and string output. Common Java types such as {@code byte}, and {@link java.lang.Integer
 * Integer} are supported.
 *
 * <p>
 * Formatters are not necessarily safe for multithreaded access. Thread safety is optional and is the responsibility of
 * users of methods in this class.
 *
 * <p>
 * Formatted printing for the Java language is heavily inspired by C's {@code printf}. Although the format strings are
 * similar to C, some customizations have been made to accommodate the Java language and exploit some of its features.
 * Also, Java formatting is more strict than C's; for example, if a conversion is incompatible with a flag, an exception
 * will be thrown. In C inapplicable flags are silently ignored. The format strings are thus intended to be recognizable
 * to C programmers but not necessarily completely compatible with those in C.
 *
 * <p>
 * Examples of expected usage:
 *
 * <blockquote>
 *
 * <pre>
 *   StringBuilder sb = new StringBuilder();
 *   // Send all output to the Appendable object sb
 *   Formatter formatter = new Formatter(sb, Locale.US);
 *
 *   // Explicit argument indices may be used to re-order output.
 *   formatter.format("%4$2s %3$2s %2$2s %1$2s", "a", "b", "c", "d")
 *   // -&gt; " d  c  b  a"
 *
 *   // The precision and width can be given to round and align the value.
 *   formatter.format("e = %+10.4f", Math.E);
 *   // -&gt; "e =    +2.7183"
 *
 *   // The '(' numeric flag may be used to format negative numbers with
 *   // parentheses rather than a minus sign.  Group separators are
 *   // automatically inserted.
 *   formatter.format("Amount gained or lost since last statement: $ %(,.2f",
 *                    balanceDelta);
 *   // -&gt; "Amount gained or lost since last statement: $ (6,217.58)"
 * </pre>
 *
 * </blockquote>
 *
 * <p>
 * Convenience methods for common formatting requests exist as illustrated by the following invocations:
 *
 * <blockquote>
 *
 * <pre>
 * // Writes formatted output to System.err.
 * System.err.printf("Unable to open file '%1$s': %2$s", fileName, exception.getMessage());
 * // -&gt; "Unable to open file 'food': No such file or directory"
 * </pre>
 *
 * </blockquote>
 *
 * <h3><a name="org">Organization</a></h3>
 *
 * <p>
 * This specification is divided into two sections. The first section, <a href="#summary">Summary</a>, covers the basic
 * formatting concepts. This section is intended for users who want to get started quickly and are familiar with
 * formatted printing in other programming languages. The second section, <a href="#detail">Details</a>, covers the
 * specific implementation details. It is intended for users who want more precise specification of formatting behavior.
 *
 * <h3><a name="summary">Summary</a></h3>
 *
 * <p>
 * This section is intended to provide a brief overview of formatting concepts. For precise behavioral details, refer to
 * the <a href="#detail">Details</a> section.
 *
 * <h4><a name="syntax">Format String Syntax</a></h4>
 *
 * <p>
 * Every method which produces formatted output requires a <i>format string</i> and an <i>argument list</i>. The format
 * string is a {@link String} which may contain fixed text and one or more embedded <i>format specifiers</i>. Consider
 * the following example:
 *
 * <blockquote>
 *
 * <pre>
 *   Calendar c = ...;
 *   String s = String.format("Hello %1$s", name);
 * </pre>
 *
 * </blockquote>
 *
 * This format string is the first argument to the {@code format} method. It contains one format specifier
 * "{@code %1$s}" which indicate how the arguments should be processed and where they should be inserted in the text.
 * The remaining portions of the format string are fixed text including {@code "Hello "} and any other spaces or
 * punctuation.
 *
 * The argument list consists of all arguments passed to the method after the format string. In the above example, the
 * argument list is of size one and consists of the {@link java.lang.String String} object {@code c}.
 *
 * <ul>
 *
 * <li>The format specifiers for general, character, and numeric types have the following syntax:
 *
 * <blockquote>
 *
 * <pre>
 *   %[argument_index$][flags][width][.precision]conversion
 * </pre>
 *
 * </blockquote>
 *
 * <p>
 * The optional <i>argument_index</i> is a decimal integer indicating the position of the argument in the argument list.
 * The first argument is referenced by "{@code 1$}", the second by "{@code 2$}", etc.
 *
 * <p>
 * The optional <i>flags</i> is a set of characters that modify the output format. The set of valid flags depends on the
 * conversion.
 *
 * <p>
 * The optional <i>width</i> is a positive decimal integer indicating the minimum number of characters to be written to
 * the output.
 *
 * <p>
 * The optional <i>precision</i> is a non-negative decimal integer usually used to restrict the number of characters.
 * The specific behavior depends on the conversion.
 *
 * <p>
 * The required <i>conversion</i> is a character indicating how the argument should be formatted. The set of valid
 * conversions for a given argument depends on the argument's data type.
 *
 * <li>The format specifiers which do not correspond to arguments have the following syntax:
 *
 * <blockquote>
 *
 * <pre>
 *   %[flags][width]conversion
 * </pre>
 *
 * </blockquote>
 *
 * <p>
 * The optional <i>flags</i> and <i>width</i> is defined as above.
 *
 * <p>
 * The required <i>conversion</i> is a character indicating content to be inserted in the output.
 *
 * </ul>
 *
 * <h4>Conversions</h4>
 *
 * <p>
 * Conversions are divided into the following categories:
 *
 * <ol>
 *
 * <li><b>General</b> - may be applied to any argument type
 *
 * <li><b>Character</b> - may be applied to basic types which represent Unicode characters: {@code char},
 * {@link Character}.
 *
 * <li><b>Numeric</b>
 *
 * <ol>
 *
 * <li><b>Integral</b> - may be applied to Java integral types: {@code byte}, {@link Byte}, {@code short}, {@link Short}
 * , {@code int} and {@link Integer}, {@code long}, and {@link Long} (but not {@code char} or {@link Character}).
 *
 * <li><b>Floating Point</b> - may be applied to Java floating-point types: {@code float}, {@link Float}, {@code double}
 * , and {@link Double}.
 *
 * </ol>
 *
 * <li><b>Percent</b> - produces a literal {@code '%'} (<tt>'&#92;u0025'</tt>)
 *
 * <li><b>Line Separator</b> - produces the platform-specific line separator
 *
 * </ol>
 *
 * <p>
 * The following table summarizes the supported conversions. Conversions denoted by an upper-case character (i.e.
 * {@code 'B'}, {@code 'H'}, {@code 'S'}, {@code 'C'}, and {@code 'X'}) are the same as those for the corresponding
 * lower-case conversion characters except that the result is converted to upper case. The result is equivalent to the
 * following invocation of {@link String#toUpperCase()}
 *
 * <pre>
 * out.toUpperCase()
 * </pre>
 *
 * <table cellpadding=5 summary="genConv">
 *
 * <tr>
 * <th valign="bottom">Conversion
 * <th valign="bottom">Argument Category
 * <th valign="bottom">Description
 *
 * <tr>
 * <td valign="top">{@code 'b'}, {@code 'B'}
 * <td valign="top">general
 * <td>If the argument <i>arg</i> is {@code null}, then the result is "{@code false}". If <i>arg</i> is a
 * {@code boolean} or {@link Boolean}, then the result is the string returned by {@link String#valueOf(boolean)
 * String.valueOf(arg)}. Otherwise, the result is "true".
 *
 * <tr>
 * <td valign="top">{@code 'h'}, {@code 'H'}
 * <td valign="top">general
 * <td>If the argument <i>arg</i> is {@code null}, then the result is "{@code null}". Otherwise, the result is obtained
 * by invoking {@code Integer.toHexString(arg.hashCode())}.
 *
 * <tr>
 * <td valign="top">{@code 's'}, {@code 'S'}
 * <td valign="top">general
 * <td>If the argument <i>arg</i> is {@code null}, then the result is "{@code null}". The result is obtained by invoking
 * {@code arg.toString()}.
 *
 * <tr>
 * <td valign="top">{@code 'c'}, {@code 'C'}
 * <td valign="top">character
 * <td>The result is a Unicode character
 *
 * <tr>
 * <td valign="top">{@code 'd'}
 * <td valign="top">integral
 * <td>The result is formatted as a decimal integer
 *
 * <tr>
 * <td valign="top">{@code 'o'}
 * <td valign="top">integral
 * <td>The result is formatted as an octal integer
 *
 * <tr>
 * <td valign="top">{@code 'x'}, {@code 'X'}
 * <td valign="top">integral
 * <td>The result is formatted as a hexadecimal integer
 *
 * <tr>
 * <td valign="top">{@code 'f'}
 * <td valign="top">floating point
 * <td>The result is formatted as a decimal number
 *
 * <tr>
 * <td valign="top">{@code '%'}
 * <td valign="top">percent
 * <td>The result is a literal {@code '%'} (<tt>'&#92;u0025'</tt>)
 *
 * <tr>
 * <td valign="top">{@code 'n'}
 * <td valign="top">line separator
 * <td>The result is the platform-specific line separator
 *
 * </table>
 *
 * <p>
 * Any characters not explicitly defined as conversions are illegal and are reserved for future extensions.
 *
 * <h4>Flags</h4>
 *
 * <p>
 * The following table summarizes the supported flags. <i>y</i> means the flag is supported for the indicated argument
 * types.
 *
 * <table cellpadding=5 summary="genConv">
 *
 * <tr>
 * <th valign="bottom">Flag
 * <th valign="bottom">General
 * <th valign="bottom">Character
 * <th valign="bottom">Integral
 * <th valign="bottom">Floating Point
 * <th valign="bottom">Description
 *
 * <tr>
 * <td>'-'
 * <td align="center" valign="top">y
 * <td align="center" valign="top">y
 * <td align="center" valign="top">y
 * <td align="center" valign="top">y
 * <td>The result will be left-justified.
 *
 * <tr>
 * <td>'#'
 * <td align="center" valign="top">y
 * <td align="center" valign="top">-
 * <td align="center" valign="top">y<sup>1</sup>
 * <td align="center" valign="top">y
 * <td>The result should use a conversion-dependent alternate form
 *
 * <tr>
 * <td>'+'
 * <td align="center" valign="top">-
 * <td align="center" valign="top">-
 * <td align="center" valign="top">y<sup>2</sup>
 * <td align="center" valign="top">y
 * <td>The result will always include a sign
 *
 * <tr>
 * <td>'&nbsp;&nbsp;'
 * <td align="center" valign="top">-
 * <td align="center" valign="top">-
 * <td align="center" valign="top">y<sup>2</sup>
 * <td align="center" valign="top">y
 * <td>The result will include a leading space for positive values
 *
 * <tr>
 * <td>'0'
 * <td align="center" valign="top">-
 * <td align="center" valign="top">-
 * <td align="center" valign="top">y
 * <td align="center" valign="top">y
 * <td>The result will be zero-padded
 *
 * <tr>
 * <td>'('
 * <td align="center" valign="top">-
 * <td align="center" valign="top">-
 * <td align="center" valign="top">y<sup>2</sup>
 * <td align="center" valign="top">y<sup>3</sup>
 * <td>The result will enclose negative numbers in parentheses
 *
 * </table>
 *
 * <p>
 * <sup>1</sup> For {@code 'o'}, {@code 'x'}, and {@code 'X'} conversions only.
 *
 * <p>
 * <sup>2</sup> For {@code 'd'} conversions applied to {@code byte}, {@link Byte}, {@code short}, {@link Short},
 * {@code int} and {@link Integer}, {@code long}, and {@link Long}.
 *
 * <p>
 * <sup>3</sup> For {@code 'f'} conversion only.
 *
 * <p>
 * Any characters not explicitly defined as flags are illegal and are reserved for future extensions.
 *
 * <h4>Width</h4>
 *
 * <p>
 * The width is the minimum number of characters to be written to the output. For the line separator conversion, width
 * is not applicable; if it is provided, an exception will be thrown.
 *
 * <h4>Precision</h4>
 *
 * <p>
 * For general argument types, the precision is the maximum number of characters to be written to the output.
 *
 * <p>
 * For the floating-point conversion {@code 'f'} the precision is the number of digits after the radix point.
 *
 * <p>
 * For character, and integral argument types and the percent and line separator conversions, the precision is not
 * applicable; if a precision is provided, an exception will be thrown.
 *
 * <h4>Argument Index</h4>
 *
 * <p>
 * The argument index is a decimal integer indicating the position of the argument in the argument list. The first
 * argument is referenced by "{@code 1$}", the second by "{@code 2$}", etc.
 *
 * <p>
 * Another way to reference arguments by position is to use the {@code '<'} (<tt>'&#92;u003c'</tt>) flag, which causes
 * the argument for the previous format specifier to be re-used. For example, the following two statements would produce
 * identical strings:
 *
 * <blockquote>
 *
 * <pre>
 *   Calendar c = ...;
 *   String s1 = String.format("Hello %1$s, how are you %1$s", name);
 *
 *   String s2 = String.format("Hello %1$s, how are you %&lt;$s", name);
 * </pre>
 *
 * </blockquote>
 *
 * <hr>
 * <h3><a name="detail">Details</a></h3>
 *
 * <p>
 * This section is intended to provide behavioral details for formatting, including conditions and exceptions, supported
 * data types, localization, and interactions between flags, conversions, and data types. For an overview of formatting
 * concepts, refer to the <a href="#summary">Summary</a>
 *
 * <p>
 * Any characters not explicitly defined as conversions, or flags are illegal and are reserved for future extensions.
 * Use of such a character in a format string will cause an {@link UnknownFormatConversionException} or
 * {@link UnknownFormatFlagsException} to be thrown.
 *
 * <p>
 * If the format specifier contains a width or precision with an invalid value or which is otherwise unsupported, then a
 * {@link IllegalFormatWidthException} or {@link IllegalFormatPrecisionException} respectively will be thrown.
 *
 * <p>
 * If a format specifier contains a conversion character that is not applicable to the corresponding argument, then an
 * {@link IllegalFormatConversionException} will be thrown.
 *
 * <p>
 * All specified exceptions may be thrown by any of the {@code format} methods of {@code Formatter}.
 *
 * <p>
 * Conversions denoted by an upper-case character (i.e. {@code 'B'}, {@code 'H'}, {@code 'S'}, {@code 'C'}, and
 * {@code 'X'}) are the same as those for the corresponding lower-case conversion characters except that the result is
 * converted to upper case. The result is equivalent to the following invocation of {@link String#toUpperCase()}
 *
 * <pre>
 * out.toUpperCase()
 * </pre>
 *
 * <h4><a name="dgen">General</a></h4>
 *
 * <p>
 * The following general conversions may be applied to any argument type:
 *
 * <table cellpadding=5 summary="dgConv">
 *
 * <tr>
 * <td valign="top">{@code 'b'}
 * <td valign="top"><tt>'&#92;u0062'</tt>
 * <td>Produces either "{@code true}" or "{@code false}" as returned by {@link Boolean#toString(boolean)}.
 *
 * <p>
 * If the argument is {@code null}, then the result is "{@code false}". If the argument is a {@code boolean} or
 * {@link Boolean}, then the result is the string returned by {@link String#valueOf(boolean) String.valueOf()}.
 * Otherwise, the result is "{@code true}".
 *
 * <p>
 * If the {@code '#'} flag is given, then a {@link FormatFlagsConversionMismatchException} will be thrown.
 *
 * <tr>
 * <td valign="top">{@code 'B'}
 * <td valign="top"><tt>'&#92;u0042'</tt>
 * <td>The upper-case variant of {@code 'b'}.
 *
 * <tr>
 * <td valign="top">{@code 'h'}
 * <td valign="top"><tt>'&#92;u0068'</tt>
 * <td>Produces a string representing the hash code value of the object.
 *
 * <p>
 * If the argument, <i>arg</i> is {@code null}, then the result is "{@code null}". Otherwise, the result is obtained by
 * invoking {@code Integer.toHexString(arg.hashCode())}.
 *
 * <p>
 * If the {@code '#'} flag is given, then a {@link FormatFlagsConversionMismatchException} will be thrown.
 *
 * <tr>
 * <td valign="top">{@code 'H'}
 * <td valign="top"><tt>'&#92;u0048'</tt>
 * <td>The upper-case variant of {@code 'h'}.
 *
 * <tr>
 * <td valign="top">{@code 's'}
 * <td valign="top"><tt>'&#92;u0073'</tt>
 * <td>Produces a string.
 *
 * <p>
 * If the argument is {@code null}, then the result is "{@code null}". The result is obtained by invoking the argument's
 * {@code toString()} method.
 *
 * <p>
 * If the {@code '#'} flag is given a {@link FormatFlagsConversionMismatchException} will be thrown.
 *
 * <tr>
 * <td valign="top">{@code 'S'}
 * <td valign="top"><tt>'&#92;u0053'</tt>
 * <td>The upper-case variant of {@code 's'}.
 *
 * </table>
 *
 * <p>
 * The following <a name="dFlags">flags</a> apply to general conversions:
 *
 * <table cellpadding=5 summary="dFlags">
 *
 * <tr>
 * <td valign="top">{@code '-'}
 * <td valign="top"><tt>'&#92;u002d'</tt>
 * <td>Left justifies the output. Spaces (<tt>'&#92;u0020'</tt>) will be added at the end of the converted value as
 * required to fill the minimum width of the field. If the width is not provided, then a
 * {@link MissingFormatWidthException} will be thrown. If this flag is not given then the output will be
 * right-justified.
 *
 * <tr>
 * <td valign="top">{@code '#'}
 * <td valign="top"><tt>'&#92;u0023'</tt>
 * <td>Requires the output use an alternate form. The definition of the form is specified by the conversion.
 *
 * </table>
 *
 * <p>
 * The <a name="genWidth">width</a> is the minimum number of characters to be written to the output. If the length of
 * the converted value is less than the width then the output will be padded by <tt>'&nbsp;&nbsp;'</tt> (
 * <tt>'&#92;u0020'</tt>) until the total number of characters equals the width. The padding is on the left by default.
 * If the {@code '-'} flag is given, then the padding will be on the right. If the width is not specified then there is
 * no minimum.
 *
 * <p>
 * The precision is the maximum number of characters to be written to the output. The precision is applied before the
 * width, thus the output will be truncated to {@code precision} characters even if the width is greater than the
 * precision. If the precision is not specified then there is no explicit limit on the number of characters.
 *
 * <h4><a name="dchar">Character</a></h4>
 *
 * This conversion may be applied to {@code char} and {@link Character}.
 *
 * <table cellpadding=5 summary="charConv">
 *
 * <tr>
 * <td valign="top">{@code 'c'}
 * <td valign="top"><tt>'&#92;u0063'</tt>
 * <td>Formats the argument as a Unicode character as described in <a href="../lang/Character.html#unicode">Unicode
 * Character Representation</a>. This may be more than one 16-bit {@code char} in the case where the argument represents
 * a supplementary character.
 *
 * <p>
 * If the {@code '#'} flag is given, then a {@link FormatFlagsConversionMismatchException} will be thrown.
 *
 * <tr>
 * <td valign="top">{@code 'C'}
 * <td valign="top"><tt>'&#92;u0043'</tt>
 * <td>The upper-case variant of {@code 'c'}.
 *
 * </table>
 *
 * <p>
 * The {@code '-'} flag defined for <a href="#dFlags">General conversions</a> applies. If the {@code '#'} flag is given,
 * then a {@link FormatFlagsConversionMismatchException} will be thrown.
 *
 * <p>
 * The width is defined as for <a href="#genWidth">General conversions</a>.
 *
 * <p>
 * The precision is not applicable. If the precision is specified then an {@link IllegalFormatPrecisionException} will
 * be thrown.
 *
 * <h4><a name="dnum">Numeric</a></h4>
 *
 * <p>
 * Numeric conversions are divided into the following categories:
 *
 * <ol>
 *
 * <li><a href="#dnint"><b>Byte, Short, Integer, and Long</b></a>
 *
 * <li><a href="#dndec"><b>Float and Double</b></a>
 *
 * </ol>
 *
 * <p>
 * Numeric types will be formatted according to the following algorithm:
 *
 * <p>
 * <b><a name="L10nAlgorithm"> Number Representation Algorithm</a></b>
 *
 * <p>
 * After digits are obtained for the integer part, fractional part, and exponent (as appropriate for the data type), the
 * following transformation is applied:
 *
 * <ol>
 *
 * <li>Each digit character <i>d</i> in the string is replaced by a digit computed relative to the zero digit
 * {@code '0'}.
 *
 * <li>If the {@code '0'} flag is given, then the zero digits {@code '0'} are inserted after the sign character, if any,
 * and before the first non-zero digit, until the length of the string is equal to the requested field width.
 *
 * <li>If the value is negative and the {@code '('} flag is given, then a {@code '('} (<tt>'&#92;u0028'</tt>) is
 * prepended and a {@code ')'} (<tt>'&#92;u0029'</tt>) is appended.
 *
 * <li>If the value is negative (or floating-point negative zero) and {@code '('} flag is not given, then a {@code '-'}
 * (<tt>'&#92;u002d'</tt>) is prepended.
 *
 * <li>If the {@code '+'} flag is given and the value is positive or zero (or floating-point positive zero), then a
 * {@code '+'} (<tt>'&#92;u002b'</tt>) will be prepended.
 *
 * </ol>
 *
 * <p>
 * If the value is NaN or positive infinity the literal strings "NaN" or "Infinity" respectively, will be output. If the
 * value is negative infinity, then the output will be "(Infinity)" if the {@code '('} flag is given otherwise the
 * output will be "-Infinity". These values are not localized.
 *
 * <p>
 * <a name="dnint"><b> Byte, Short, Integer, and Long </b></a>
 *
 * <p>
 * The following conversions may be applied to {@code byte}, {@link Byte}, {@code short}, {@link Short}, {@code int} and
 * {@link Integer}, {@code long}, and {@link Long}.
 *
 * <table cellpadding=5 summary="IntConv">
 *
 * <tr>
 * <td valign="top">{@code 'd'}
 * <td valign="top"><tt>'&#92;u0064'</tt>
 * <td>Formats the argument as a decimal integer. The <a href="#L10nAlgorithm">localization algorithm</a> is applied.
 *
 * <p>
 * If the {@code '0'} flag is given and the value is negative, then the zero padding will occur after the sign.
 *
 * <p>
 * If the {@code '#'} flag is given then a {@link FormatFlagsConversionMismatchException} will be thrown.
 *
 * <tr>
 * <td valign="top">{@code 'o'}
 * <td valign="top"><tt>'&#92;u006f'</tt>
 * <td>Formats the argument as an integer in base eight. No localization is applied.
 *
 * <p>
 * If <i>x</i> is negative then the result will be an unsigned value generated by adding 2<sup>n</sup> to the value
 * where {@code n} is the number of bits in the type as returned by the static {@code SIZE} field in the
 * {@linkplain Byte#SIZE Byte}, {@linkplain Short#SIZE Short}, {@linkplain Integer#SIZE Integer}, or
 * {@linkplain Long#SIZE Long} classes as appropriate.
 *
 * <p>
 * If the {@code '#'} flag is given then the output will always begin with the radix indicator {@code '0'}.
 *
 * <p>
 * If the {@code '0'} flag is given then the output will be padded with leading zeros to the field width following any
 * indication of sign.
 *
 * <p>
 * If {@code '('}, {@code '+'}, or '&nbsp;&nbsp;' flags are given then a {@link FormatFlagsConversionMismatchException}
 * will be thrown.
 *
 * <tr>
 * <td valign="top">{@code 'x'}
 * <td valign="top"><tt>'&#92;u0078'</tt>
 * <td>Formats the argument as an integer in base sixteen. No localization is applied.
 *
 * <p>
 * If <i>x</i> is negative then the result will be an unsigned value generated by adding 2<sup>n</sup> to the value
 * where {@code n} is the number of bits in the type as returned by the static {@code SIZE} field in the
 * {@linkplain Byte#SIZE Byte}, {@linkplain Short#SIZE Short}, {@linkplain Integer#SIZE Integer}, or
 * {@linkplain Long#SIZE Long} classes as appropriate.
 *
 * <p>
 * If the {@code '#'} flag is given then the output will always begin with the radix indicator {@code "0x"}.
 *
 * <p>
 * If the {@code '0'} flag is given then the output will be padded to the field width with leading zeros after the radix
 * indicator or sign (if present).
 *
 * <p>
 * If {@code '('}, <tt>'&nbsp;&nbsp;'</tt>, or {@code '+'} flags are given then a
 * {@link FormatFlagsConversionMismatchException} will be thrown.
 *
 * <tr>
 * <td valign="top">{@code 'X'}
 * <td valign="top"><tt>'&#92;u0058'</tt>
 * <td>The upper-case variant of {@code 'x'}. The entire string representing the number will be converted to
 * {@linkplain String#toUpperCase upper case} including the {@code 'x'} (if any) and all hexadecimal digits -
 * {@code 'f'} (<tt>'&#92;u0066'</tt>).
 *
 * </table>
 *
 * <p>
 * If the conversion is {@code 'o'}, {@code 'x'}, or {@code 'X'} and both the {@code '#'} and the {@code '0'} flags are
 * given, then result will contain the radix indicator ({@code '0'} for octal and {@code "0x"} or {@code "0X"} for
 * hexadecimal), some number of zeros (based on the width), and the value.
 *
 * <p>
 * If the {@code '-'} flag is not given, then the space padding will occur before the sign.
 *
 * <p>
 * The following <a name="intFlags">flags</a> apply to numeric integral conversions:
 *
 * <table cellpadding=5 summary="intFlags">
 *
 * <tr>
 * <td valign="top">{@code '+'}
 * <td valign="top"><tt>'&#92;u002b'</tt>
 * <td>Requires the output to include a positive sign for all positive numbers. If this flag is not given then only
 * negative values will include a sign.
 *
 * <p>
 * If both the {@code '+'} and <tt>'&nbsp;&nbsp;'</tt> flags are given then an {@link IllegalFormatFlagsException} will
 * be thrown.
 *
 * <tr>
 * <td valign="top"><tt>'&nbsp;&nbsp;'</tt>
 * <td valign="top"><tt>'&#92;u0020'</tt>
 * <td>Requires the output to include a single extra space (<tt>'&#92;u0020'</tt>) for non-negative values.
 *
 * <p>
 * If both the {@code '+'} and <tt>'&nbsp;&nbsp;'</tt> flags are given then an {@link IllegalFormatFlagsException} will
 * be thrown.
 *
 * <tr>
 * <td valign="top">{@code '0'}
 * <td valign="top"><tt>'&#92;u0030'</tt>
 * <td>Requires the output to be padded with leading {@code '0'} to the minimum field width following any sign or radix
 * indicator except when converting NaN or infinity. If the width is not provided, then a
 * {@link MissingFormatWidthException} will be thrown.
 *
 * <p>
 * If both the {@code '-'} and {@code '0'} flags are given then an {@link IllegalFormatFlagsException} will be thrown.
 *
 * <tr>
 * <td valign="top">{@code ','}
 * <td valign="top"><tt>'&#92;u002c'</tt>
 * <td>Requires the output to include the group separators as described in the <a href="#L10nGroup">"group" section</a>
 * of the localization algorithm.
 *
 * <tr>
 * <td valign="top">{@code '('}
 * <td valign="top"><tt>'&#92;u0028'</tt>
 * <td>Requires the output to prepend a {@code '('} (<tt>'&#92;u0028'</tt>) and append a {@code ')'} (
 * <tt>'&#92;u0029'</tt>) to negative values.
 *
 * </table>
 *
 * <p>
 * If no <a name="intdFlags">flags</a> are given the default formatting is as follows:
 *
 * <ul>
 *
 * <li>The output is right-justified within the {@code width}
 *
 * <li>Negative numbers begin with a {@code '-'} (<tt>'&#92;u002d'</tt>)
 *
 * <li>Positive numbers and zero do not include a sign or extra leading space
 *
 * <li>No grouping separators are included
 *
 * </ul>
 *
 * <p>
 * The <a name="intWidth">width</a> is the minimum number of characters to be written to the output. This includes any
 * signs, digits, grouping separators, radix indicator, and parentheses. If the length of the converted value is less
 * than the width then the output will be padded by spaces (<tt>'&#92;u0020'</tt>) until the total number of characters
 * equals width. The padding is on the left by default. If {@code '-'} flag is given then the padding will be on the
 * right. If width is not specified then there is no minimum.
 *
 * <p>
 * The precision is not applicable. If precision is specified then an {@link IllegalFormatPrecisionException} will be
 * thrown.
 *
 * <p>
 * <a name="dndec"><b> Float and Double</b></a>
 *
 * <p>
 * The following conversions may be applied to {@code float}, {@link Float}, {@code double} and {@link Double}.
 *
 * <table cellpadding=5 summary="floatConv">
 *
 * <tr>
 * <td valign="top">{@code 'f'}
 * <td valign="top"><tt>'&#92;u0066'</tt>
 * <td>Requires the output to be formatted using <a name="decimal">decimal format</a>. The
 * <a href="#L10nAlgorithm">localization algorithm</a> is applied.
 *
 * <p>
 * The result is a string that represents the sign and magnitude (absolute value) of the argument. The formatting of the
 * sign is described in the <a href="#L10nAlgorithm">localization algorithm</a>. The formatting of the magnitude
 * <i>m</i> depends upon its value.
 *
 * <p>
 * If <i>m</i> NaN or infinite, the literal strings "NaN" or "Infinity", respectively, will be output. These values are
 * not localized.
 *
 * <p>
 * The magnitude is formatted as the integer part of <i>m</i>, with no leading zeroes, followed by the decimal separator
 * followed by one or more decimal digits representing the fractional part of <i>m</i>.
 *
 * <p>
 * The number of digits in the result for the fractional part of <i>m</i> or <i>a</i> is equal to the precision. If the
 * precision is not specified then the default value is {@code 6}. If the precision is less than the number of digits
 * which would appear after the decimal point in the string returned by {@link Float#toString(float)} or
 * {@link Double#toString(double)} respectively, then the result cannot be printed and a
 * {@link IllegalFormatConversionException} is thrown. Otherwise, zeros may be appended to reach the precision. For a
 * canonical representation of the value, use {@link Float#toString(float)} or {@link Double#toString(double)} as
 * appropriate.
 *
 * </table>
 *
 * <p>
 * All <a href="#intFlags">flags</a> defined for Byte, Short, Integer, and Long apply.
 *
 * <p>
 * If the {@code '#'} flag is given, then the decimal separator will always be present.
 *
 * <p>
 * If no <a name="floatdFlags">flags</a> are given the default formatting is as follows:
 *
 * <ul>
 *
 * <li>The output is right-justified within the {@code width}
 *
 * <li>Negative numbers begin with a {@code '-'}
 *
 * <li>Positive numbers and positive zero do not include a sign or extra leading space
 *
 * <li>No grouping separators are included
 *
 * <li>The decimal separator will only appear if a digit follows it
 *
 * </ul>
 *
 * <p>
 * The <a name="floatDWidth">width</a> is the minimum number of characters to be written to the output. This includes
 * any signs, digits, grouping separators, decimal separators, exponential symbol, radix indicator, parentheses, and
 * strings representing infinity and NaN as applicable. If the length of the converted value is less than the width then
 * the output will be padded by spaces (<tt>'&#92;u0020'</tt>) until the total number of characters equals width. The
 * padding is on the left by default. If the {@code '-'} flag is given then the padding will be on the right. If width
 * is not specified then there is no minimum.
 *
 * <p>
 * If the <a name="floatDPrec">conversion</a> is {@code 'f'}, then the precision is the number of digits after the
 * decimal separator. If the precision is not specified, then it is assumed to be {@code 6}.
 *
 * <p>
 * If the conversion is {@code 'g'} or {@code 'G'}, then the precision is the total number of significant digits in the
 * resulting magnitude after rounding. If the precision is not specified, then the default value is {@code 6}. If the
 * precision is {@code 0}, then it is taken to be {@code 1}.
 *
 * <h4><a name="dper">Percent</a></h4>
 *
 * <p>
 * The conversion does not correspond to any argument.
 *
 * <table cellpadding=5 summary="DTConv">
 *
 * <tr>
 * <td valign="top">{@code '%'}
 * <td>The result is a literal {@code '%'} (<tt>'&#92;u0025'</tt>)
 *
 * <p>
 * The width is the minimum number of characters to be written to the output including the {@code '%'}. If the length of
 * the converted value is less than the {@code width} then the output will be padded by spaces (<tt>'&#92;u0020'</tt>)
 * until the total number of characters equals width. The padding is on the left. If width is not specified then just
 * the {@code '%'} is output.
 *
 * <p>
 * The {@code '-'} flag defined for <a href="#dFlags">General conversions</a> applies. If any other flags are provided,
 * then a {@link FormatFlagsConversionMismatchException} will be thrown.
 *
 * <p>
 * The precision is not applicable. If the precision is specified an {@link IllegalFormatPrecisionException} will be
 * thrown.
 *
 * </table>
 *
 * <h4><a name="dls">Line Separator</a></h4>
 *
 * <p>
 * The conversion does not correspond to any argument.
 *
 * <table cellpadding=5 summary="DTConv">
 *
 * <tr>
 * <td valign="top">{@code 'n'}
 * <td>the platform-specific line separator as returned by {@link System#getProperty
 * System.getProperty("line.separator")}.
 *
 * </table>
 *
 * <p>
 * Flags, width, and precision are not applicable. If any are provided an {@link IllegalFormatFlagsException},
 * {@link IllegalFormatWidthException}, and {@link IllegalFormatPrecisionException}, respectively will be thrown.
 *
 * <h4><a name="dpos">Argument Index</a></h4>
 *
 * <p>
 * Format specifiers can reference arguments in three ways:
 *
 * <ul>
 *
 * <li><i>Explicit indexing</i> is used when the format specifier contains an argument index. The argument index is a
 * decimal integer indicating the position of the argument in the argument list. The first argument is referenced by
 * "{@code 1$}", the second by "{@code 2$}", etc. An argument may be referenced more than once.
 *
 * <p>
 * For example:
 *
 * <blockquote>
 *
 * <pre>
 * formatter.format("%4$s %3$s %2$s %1$s %4$s %3$s %2$s %1$s", "a", "b", "c", "d")
 * // -&gt; "d c b a d c b a"
 * </pre>
 *
 * </blockquote>
 *
 * <li><i>Relative indexing</i> is used when the format specifier contains a {@code '<'} (<tt>'&#92;u003c'</tt>) flag
 * which causes the argument for the previous format specifier to be re-used. If there is no previous argument, then a
 * {@link MissingFormatArgumentException} is thrown.
 *
 * <blockquote>
 *
 * <pre>
 * formatter.format("%s %s %&lt;s %&lt;s", "a", "b", "c", "d")
 * // -&gt; "a b b b"
 * // "c" and "d" are ignored because they are not referenced
 * </pre>
 *
 * </blockquote>
 *
 * <li><i>Ordinary indexing</i> is used when the format specifier contains neither an argument index nor a {@code '<'}
 * flag. Each format specifier which uses ordinary indexing is assigned a sequential implicit index into argument list
 * which is independent of the indices used by explicit or relative indexing.
 *
 * <blockquote>
 *
 * <pre>
 * formatter.format("%s %s %s %s", "a", "b", "c", "d")
 * // -&gt; "a b c d"
 * </pre>
 *
 * </blockquote>
 *
 * </ul>
 *
 * <p>
 * It is possible to have a format string which uses all forms of indexing, for example:
 *
 * <blockquote>
 *
 * <pre>
 * formatter.format("%2$s %s %&lt;s %s", "a", "b", "c", "d")
 * // -&gt; "b a a b"
 * // "c" and "d" are ignored because they are not referenced
 * </pre>
 *
 * </blockquote>
 *
 * <p>
 * The maximum number of arguments is limited by the maximum dimension of a Java array as defined by <cite>The
 * Java&trade; Virtual Machine Specification</cite>. If the argument index is does not correspond to an available
 * argument, then a {@link MissingFormatArgumentException} is thrown.
 *
 * <p>
 * If there are more arguments than format specifiers, the extra arguments are ignored.
 *
 * <p>
 * Unless otherwise specified, passing a {@code null} argument to any method or constructor in this class will cause a
 * {@link NullPointerException} to be thrown.
 *
 * @author Iris Clark
 * @since 1.5
 */
public final class Formatter implements Closeable, Flushable {
	@Nullable
	private Appendable a;

	@Nullable
	private IOException lastException;

	private static final Appendable nonNullAppendable(@Nullable Appendable a) {
		if (a == null) {
			return new StringBuilder();
		}

		return a;
	}

	/**
	 * Constructs a new formatter.
	 *
	 * <p>
	 * The destination of the formatted output is a {@link StringBuilder} which may be retrieved by invoking {@link #out
	 * out()} and whose current content may be converted into a string by invoking {@link #toString toString()}.
	 */
	public Formatter() {
		this(new StringBuilder());
	}

	/**
	 * Constructs a new formatter with the specified destination.
	 *
	 * @param a
	 *            Destination for the formatted output. If {@code a} is {@code null} then a {@link StringBuilder} will
	 *            be created.
	 */
	public Formatter(@Nullable Appendable a) {
		this.a = nonNullAppendable(a);
	}

	/**
	 * Constructs a new formatter with the specified print stream.
	 *
	 * <p>
	 * Characters are written to the given {@link java.io.PrintStream PrintStream} object and are therefore encoded
	 * using that object's charset.
	 *
	 * @param ps
	 *            The stream to use as the destination of this formatter.
	 */
	public Formatter(PrintStream ps) {
		this((Appendable) Objects.requireNonNull(ps));
	}

	/**
	 * Constructs a new formatter with the specified output stream.
	 *
	 * <p>
	 * The charset used is the default charset for this instance of the Java virtual machine.
	 *
	 * @param os
	 *            The output stream to use as the destination of this formatter. The output will be buffered.
	 */
	public Formatter(OutputStream os) {
		this(new BufferedWriter(new OutputStreamWriter(Objects.requireNonNull(os))));
	}

	/**
	 * Constructs a new formatter with the specified output stream and charset.
	 *
	 * @param os
	 *            The output stream to use as the destination of this formatter. The output will be buffered.
	 *
	 * @param csn
	 *            The name of a supported charset.
	 *
	 * @throws UnsupportedEncodingException
	 *             If the named charset is not supported.
	 */
	public Formatter(OutputStream os, String csn) throws UnsupportedEncodingException {
		this(new BufferedWriter(new OutputStreamWriter(Objects.requireNonNull(os), csn)));
	}

	/**
	 * Returns the destination for the output.
	 *
	 * @return The destination for the output
	 *
	 * @throws FormatterClosedException
	 *             If this formatter has been closed by invoking its {@link #close()} method
	 */
	public Appendable out() {
		return ensureOpen();
	}

	/**
	 * Returns the result of invoking {@code toString()} on the destination for the output. For example, the following
	 * code formats text into a {@link StringBuilder} then retrieves the resultant string:
	 *
	 * <blockquote>
	 *
	 * <pre>
	 * Formatter f = new Formatter();
	 * f.format("Last reboot at %tc", lastRebootDate);
	 * String s = f.toString();
	 * // -&gt; s == "Last reboot at Sat Jan 01 00:00:00 PST 2000"
	 * </pre>
	 *
	 * </blockquote>
	 *
	 * <p>
	 * An invocation of this method behaves in exactly the same way as the invocation
	 *
	 * <pre>
	 * out().toString()
	 * </pre>
	 *
	 * <p>
	 * Depending on the specification of {@code toString} for the {@link Appendable}, the returned string may or may not
	 * contain the characters written to the destination. For instance, buffers typically return their contents in
	 * {@code toString()}, but streams cannot since the data is discarded.
	 *
	 * @return The result of invoking {@code toString()} on the destination for the output
	 *
	 * @throws FormatterClosedException
	 *             If this formatter has been closed by invoking its {@link #close()} method
	 */
	@Override
	public String toString() {
		Appendable a = ensureOpen();
		String aString = a.toString();
		assert aString != null; // (required because of Eclipse false positive null access detected)
		return aString;
	}

	/**
	 * Flushes this formatter. If the destination implements the {@link java.io.Flushable} interface, its {@code flush}
	 * method will be invoked.
	 *
	 * <p>
	 * Flushing a formatter writes any buffered output in the destination to the underlying stream.
	 *
	 * @throws FormatterClosedException
	 *             If this formatter has been closed by invoking its {@link #close()} method
	 */
	@Override
	public void flush() {
		Appendable a = ensureOpen();
		if (a instanceof Flushable) {
			try {
				((Flushable) a).flush();
			} catch (IOException ioe) {
				this.lastException = ioe;
			}
		}
	}

	/**
	 * Closes this formatter. If the destination implements the {@link java.io.Closeable} interface, its {@code close}
	 * method will be invoked.
	 *
	 * <p>
	 * Closing a formatter allows it to release resources it may be holding (such as open files). If the formatter is
	 * already closed, then invoking this method has no effect.
	 *
	 * <p>
	 * Attempting to invoke any methods except {@link #ioException()} in this formatter after it has been closed will
	 * result in a {@link FormatterClosedException}.
	 */
	@Override
	public void close() {
		Appendable a = this.a;
		if (a == null) {
			return;
		}
		try {
			if (a instanceof Closeable) {
				((Closeable) a).close();
			}
		} catch (IOException ioe) {
			this.lastException = ioe;
		} finally {
			this.a = null;
		}
	}

	private Appendable ensureOpen() {
		Appendable a = this.a;
		if (a == null) {
			throw new FormatterClosedException();
		}
		return a;
	}

	/**
	 * Returns the {@code IOException} last thrown by this formatter's {@link Appendable}.
	 *
	 * <p>
	 * If the destination's {@code append()} method never throws {@code IOException}, then this method will always
	 * return {@code null}.
	 *
	 * @return The last exception thrown by the Appendable or {@code null} if no such exception exists.
	 */
	@Nullable
	public IOException ioException() {
		return this.lastException;
	}

	/**
	 * Writes a formatted string to this object's destination using the specified format string and arguments. The
	 * locale used is the one defined during the construction of this formatter.
	 *
	 * @param format
	 *            A format string as described in <a href="#syntax">Format string syntax</a>.
	 *
	 * @param args
	 *            Arguments referenced by the format specifiers in the format string. If there are more arguments than
	 *            format specifiers, the extra arguments are ignored. The maximum number of arguments is limited by the
	 *            maximum dimension of a Java array as defined by <cite>The Java&trade; Virtual Machine
	 *            Specification</cite>.
	 *
	 * @throws IllegalFormatException
	 *             If a format string contains an illegal syntax, a format specifier that is incompatible with the given
	 *             arguments, insufficient arguments given the format string, or other illegal conditions. For
	 *             specification of all possible formatting errors, see the <a href="#detail">Details</a> section of the
	 *             formatter class specification.
	 *
	 * @throws FormatterClosedException
	 *             If this formatter has been closed by invoking its {@link #close()} method
	 *
	 * @return This formatter
	 */
	public Formatter format(String format, @Nullable Object... args) {
		ensureOpen();

		// index of last argument referenced
		int last = -1;
		// last ordinary index
		int lasto = -1;

		FormatString[] fsa = parse(format);
		for (int i = 0; i < fsa.length; i++) {
			FormatString fs = fsa[i];
			int index = fs.index();
			try {
				switch (index) {
				case -2: // fixed string, "%n", or "%%"
					fs.print(null);
					break;
				case -1: // relative index
					if (last < 0 || (args != null && last > args.length - 1)) {
						throw new MissingFormatArgumentException(fs.toString());
					}
					fs.print((args == null ? null : args[last]));
					break;
				case 0: // ordinary index
					lasto++;
					last = lasto;
					if (args != null && lasto > args.length - 1) {
						throw new MissingFormatArgumentException(fs.toString());
					}
					fs.print((args == null ? null : args[lasto]));
					break;
				default: // explicit index
					last = index - 1;
					if (args != null && last > args.length - 1) {
						throw new MissingFormatArgumentException(fs.toString());
					}
					fs.print((args == null ? null : args[last]));
					break;
				}
			} catch (IOException x) {
				this.lastException = x;
			}
		}
		return this;
	}

	/**
	 * Finds format specifiers in the format string.
	 * <p>
	 * %[argument_index$][flags][width][.precision]conversion
	 */
	private FormatString[] parse(String s) {
		List<FormatString> parts = new ArrayList<>();
		int lastPosition = 0;
		while (true) {
			int formatIndex = s.indexOf('%', lastPosition);
			if (formatIndex == -1) {
				if (lastPosition < s.length()) {
					// Add last piece of string.
					FixedString fixedString = new FixedString(s.substring(lastPosition, s.length()));
					parts.add(fixedString);
				}
				break;
			}
			if (lastPosition != formatIndex) {
				// Add piece of string between two format.
				FixedString fixedString = new FixedString(s.substring(lastPosition, formatIndex));
				parts.add(fixedString);
			}

			// Parse format.
			int formatEndIndex = parseFormat(s, formatIndex, parts);

			lastPosition = formatEndIndex + 1;
		}
		// throw new UnknownFormatConversionException("");
		return parts.toArray(new FormatString[parts.size()]);
	}

	private int parseFormat(String s, int formatIndex, List<FormatString> parts) {
		int argumentIndex = 0;
		Flags flags = new Flags(0);
		int width = -1;
		int precision = -1;
		// 0: argument index
		// 1: flags
		// 2: width
		// 3: precision
		// 4: conversion
		int state = 0;
		int number = -1;
		while (true) {
			if (++formatIndex >= s.length()) {
				throw new UnknownFormatConversionException(s);
			}
			char c = s.charAt(formatIndex);
			if (c > '0' && c < '9') {
				int lastNumberIndex = searchLastNumberIndex(s, formatIndex);
				String numberString = s.substring(formatIndex, lastNumberIndex);
				number = Integer.parseInt(numberString);
				formatIndex = lastNumberIndex - 1;
			} else if (state == 0 && c == '$') {
				if (number <= 0) {
					throw new UnknownFormatConversionException(s);
				}
				argumentIndex = number;
				number = -1;
				state = 1;
			} else if (state <= 1 && isFlag(c)) {
				Flags newFlags = Flags.parse(c);
				if (flags.contains(newFlags)) {
					throw new DuplicateFormatFlagsException(newFlags.toString());
				}
				flags.add(newFlags);
			} else if (state <= 2 && number != -1) {
				width = number;
				number = -1;
				state = 3;
				formatIndex--;
			} else if (state <= 3 && c == '.') {
				try {
					int lastNumberIndex = searchLastNumberIndex(s, formatIndex);
					String numberString = s.substring(formatIndex + 1, lastNumberIndex);
					precision = Integer.parseInt(numberString);
					formatIndex = lastNumberIndex - 1;
					state = 4;
				} catch (NumberFormatException | IndexOutOfBoundsException e) {
					throw new UnknownFormatConversionException(s);
				}
			} else if (state <= 4) {
				if (!Conversion.isValid(c)) {
					throw new UnknownFormatConversionException(String.valueOf(c));
				}
				if (Character.isUpperCase(c)) {
					flags.add(Flags.UPPERCASE);
				}
				char conversion = Character.toLowerCase(c);
				FormatSpecifier formatSpecifier = new FormatSpecifier(argumentIndex, flags, width, precision,
						conversion);
				parts.add(formatSpecifier);
				return formatIndex;
			}
		}
	}

	private boolean isFlag(char c) {
		return c == '-' || c == '#' || c == '+' || c == ' ' || c == '0' || c == '(' || c == '<';
	}

	private int searchLastNumberIndex(String s, int numberIndex) {
		while (true) {
			char c = s.charAt(++numberIndex);
			if (c < '0' || c > '9') {
				return numberIndex;
			}
		}
	}

	private interface FormatString {
		int index();

		void print(@Nullable Object arg) throws IOException;

		@Override
		String toString();
	}

	private class FixedString implements FormatString {
		private final String s;

		FixedString(String s) {
			this.s = s;
		}

		@Override
		public int index() {
			return -2;
		}

		@Override
		public void print(@Nullable Object arg) throws IOException {
			Appendable a = Formatter.this.a;
			assert a != null;// ensureOpen() by Formatter.format(...)
			a.append(this.s);
		}

		@Override
		public String toString() {
			return this.s;
		}
	}

	private class FormatSpecifier implements FormatString {
		private int index = -1;
		private Flags f = Flags.NONE;
		private final int width;
		private final int precision;
		private final char c;

		@Override
		public int index() {
			return this.index;
		}

		FormatSpecifier(int argumentIndex, Flags flags, int width, int precision, char conversion) {
			this.index = argumentIndex;
			this.f = flags;
			if (this.f.contains(Flags.PREVIOUS)) {
				this.index = -1;
			}
			this.width = width;
			this.precision = precision;
			this.c = conversion;
			if (Conversion.isText(conversion)) {
				this.index = -2;
			}

			if (Conversion.isGeneral(this.c)) {
				checkGeneral();
			} else if (Conversion.isCharacter(this.c)) {
				checkCharacter();
			} else if (Conversion.isInteger(this.c)) {
				checkInteger();
			} else if (Conversion.isFloat(this.c)) {
				checkFloat();
			} else if (Conversion.isText(this.c)) {
				checkText();
			} else {
				throw new UnknownFormatConversionException(String.valueOf(this.c));
			}
		}

		@Override
		public void print(@Nullable Object arg) throws IOException {
			Appendable a = Formatter.this.a;
			assert a != null; // ensureOpen() by Formatter.format(...)
			switch (this.c) {
			case Conversion.DECIMAL_INTEGER:
			case Conversion.OCTAL_INTEGER:
			case Conversion.HEXADECIMAL_INTEGER:
				printInteger(arg);
				break;
			// case Conversion.SCIENTIFIC:
			// case Conversion.GENERAL:
			case Conversion.DECIMAL_FLOAT:
				// case Conversion.HEXADECIMAL_FLOAT:
				printFloat(arg);
				break;
			case Conversion.CHARACTER:
			case Conversion.CHARACTER_UPPER:
				printCharacter(arg);
				break;
			case Conversion.BOOLEAN:
				printBoolean(arg);
				break;
			case Conversion.STRING:
				printString(arg);
				break;
			case Conversion.HASHCODE:
				printHashCode(arg);
				break;
			case Conversion.LINE_SEPARATOR:
				a.append(System.getProperty("line.separator")); //$NON-NLS-1$
				break;
			case Conversion.PERCENT_SIGN:
				a.append('%');
				break;
			default:
				assert false;
			}
		}

		private void printInteger(@Nullable Object arg) throws IOException {
			if (arg == null) {
				print("null"); //$NON-NLS-1$
			} else if (arg instanceof Byte) {
				print(((Byte) arg).byteValue());
			} else if (arg instanceof Short) {
				print(((Short) arg).shortValue());
			} else if (arg instanceof Integer) {
				print(((Integer) arg).intValue());
			} else if (arg instanceof Long) {
				print(((Long) arg).longValue());
			} else {
				failConversion(this.c, arg);
			}
		}

		private void printFloat(@Nullable Object arg) throws IOException {
			if (arg == null) {
				print("null"); //$NON-NLS-1$
			} else if (arg instanceof Float) {
				print(((Float) arg).floatValue(), arg);
			} else if (arg instanceof Double) {
				print(((Double) arg).doubleValue(), arg);
			} else {
				failConversion(this.c, arg);
			}
		}

		private void printCharacter(@Nullable Object arg) throws IOException {
			if (arg == null) {
				print("null"); //$NON-NLS-1$
				return;
			}
			String s = null;
			if (arg instanceof Character) {
				s = ((Character) arg).toString();
			} else {
				failConversion(this.c, arg);
			}
			assert s != null;
			print(s);
		}

		private void printString(@Nullable Object arg) throws IOException {
			if (this.f.contains(Flags.ALTERNATE)) {
				failMismatch(Flags.ALTERNATE, 's');
			}
			if (arg == null) {
				print("null"); //$NON-NLS-1$
			} else {
				String argString = arg.toString();
				assert argString != null; // (required because of Eclipse false positive null access detected)
				print(argString);
			}
		}

		private void printBoolean(@Nullable Object arg) throws IOException {
			String s;
			if (arg != null) {
				s = ((arg instanceof Boolean) ? ((Boolean) arg).toString() : Boolean.toString(true));
			} else {
				s = Boolean.toString(false);
			}
			print(s);
		}

		private void printHashCode(@Nullable Object arg) throws IOException {
			String s = (arg == null ? "null" : Integer.toHexString(arg.hashCode())); //$NON-NLS-1$
			print(s);
		}

		private void print(String s) throws IOException {
			if (this.precision != -1 && this.precision < s.length()) {
				s = s.substring(0, this.precision);
			}
			if (this.f.contains(Flags.UPPERCASE)) {
				s = s.toUpperCase();
			}
			Appendable a = Formatter.this.a;
			assert a != null; // already ensured in FormatSpecifier.print(...)
			a.append(justify(s));
		}

		private String justify(String s) {
			if (this.width == -1) {
				return s;
			}
			StringBuilder sb = new StringBuilder();
			boolean pad = this.f.contains(Flags.LEFT_JUSTIFY);
			int sp = this.width - s.length();
			if (!pad) {
				for (int i = 0; i < sp; i++) {
					sb.append(' ');
				}
			}
			sb.append(s);
			if (pad) {
				for (int i = 0; i < sp; i++) {
					sb.append(' ');
				}
			}
			return sb.toString();
		}

		@Override
		public String toString() {
			StringBuilder sb = new StringBuilder("%"); //$NON-NLS-1$
			// Flags.UPPERCASE is set internally for legal conversions.
			Flags dupf = this.f.dup().remove(Flags.UPPERCASE);
			sb.append(dupf.toString());
			if (this.index > 0) {
				sb.append(this.index).append('$');
			}
			if (this.width != -1) {
				sb.append(this.width);
			}
			if (this.precision != -1) {
				sb.append('.').append(this.precision);
			}
			sb.append(this.f.contains(Flags.UPPERCASE) ? Character.toUpperCase(this.c) : this.c);
			return sb.toString();
		}

		private void checkGeneral() {
			if ((this.c == Conversion.BOOLEAN || this.c == Conversion.HASHCODE) && this.f.contains(Flags.ALTERNATE)) {
				failMismatch(Flags.ALTERNATE, this.c);
			}
			// '-' requires a width
			if (this.width == -1 && this.f.contains(Flags.LEFT_JUSTIFY)) {
				throw new MissingFormatWidthException(toString());
			}
			checkBadFlags(Flags.PLUS, Flags.LEADING_SPACE, Flags.ZERO_PAD, Flags.PARENTHESES);
		}

		private void checkCharacter() {
			if (this.precision != -1) {
				throw new IllegalFormatPrecisionException(this.precision);
			}
			checkBadFlags(Flags.ALTERNATE, Flags.PLUS, Flags.LEADING_SPACE, Flags.ZERO_PAD, Flags.PARENTHESES);
			// '-' requires a width
			if (this.width == -1 && this.f.contains(Flags.LEFT_JUSTIFY)) {
				throw new MissingFormatWidthException(toString());
			}
		}

		private void checkInteger() {
			checkNumeric();
			if (this.precision != -1) {
				throw new IllegalFormatPrecisionException(this.precision);
			}

			if (this.c == Conversion.DECIMAL_INTEGER) {
				checkBadFlags(Flags.ALTERNATE);
			}
		}

		private void checkBadFlags(Flags... badFlags) {
			for (int i = 0; i < badFlags.length; i++) {
				Flags e = badFlags[i];
				assert e != null; // this private method is never called with null arguments
				if (this.f.contains(e)) {
					failMismatch(e, this.c);
				}
			}
		}

		private void checkFloat() {
			checkNumeric();
		}

		private void checkNumeric() {
			if (this.width != -1 && this.width < 0) {
				throw new IllegalFormatWidthException(this.width);
			}

			if (this.precision != -1 && this.precision < 0) {
				throw new IllegalFormatPrecisionException(this.precision);
			}

			// '-' and '0' require a width
			if (this.width == -1 && (this.f.contains(Flags.LEFT_JUSTIFY) || this.f.contains(Flags.ZERO_PAD))) {
				throw new MissingFormatWidthException(toString());
			}

			// bad combination
			if ((this.f.contains(Flags.PLUS) && this.f.contains(Flags.LEADING_SPACE))
					|| (this.f.contains(Flags.LEFT_JUSTIFY) && this.f.contains(Flags.ZERO_PAD))) {
				throw new IllegalFormatFlagsException(this.f.toString());
			}
		}

		private void checkText() {
			if (this.precision != -1) {
				throw new IllegalFormatPrecisionException(this.precision);
			}
			switch (this.c) {
			case Conversion.PERCENT_SIGN:
				if (this.f.valueOf() != Flags.LEFT_JUSTIFY.valueOf() && this.f.valueOf() != Flags.NONE.valueOf()) {
					throw new IllegalFormatFlagsException(this.f.toString());
				}
				// '-' requires a width
				if (this.width == -1 && this.f.contains(Flags.LEFT_JUSTIFY)) {
					throw new MissingFormatWidthException(toString());
				}
				break;
			case Conversion.LINE_SEPARATOR:
				if (this.width != -1) {
					throw new IllegalFormatWidthException(this.width);
				}
				if (this.f.valueOf() != Flags.NONE.valueOf()) {
					throw new IllegalFormatFlagsException(this.f.toString());
				}
				break;
			default:
				assert false;
			}
		}

		private void print(byte value) throws IOException {
			long v = value;
			if (value < 0 && (this.c == Conversion.OCTAL_INTEGER || this.c == Conversion.HEXADECIMAL_INTEGER)) {
				v += (1L << 8);
				assert v >= 0 : v;
			}
			print(v);
		}

		private void print(short value) throws IOException {
			long v = value;
			if (value < 0 && (this.c == Conversion.OCTAL_INTEGER || this.c == Conversion.HEXADECIMAL_INTEGER)) {
				v += (1L << 16);
				assert v >= 0 : v;
			}
			print(v);
		}

		private void print(int value) throws IOException {
			long v = value;
			if (value < 0 && (this.c == Conversion.OCTAL_INTEGER || this.c == Conversion.HEXADECIMAL_INTEGER)) {
				v += (1L << 32);
				assert v >= 0 : v;
			}
			print(v);
		}

		private void print(long value) throws IOException {

			StringBuilder sb = new StringBuilder();

			if (this.c == Conversion.DECIMAL_INTEGER) {
				boolean neg = value < 0;
				char[] va;
				if (value < 0) {
					va = Long.toString(value, 10).substring(1).toCharArray();
				} else {
					va = Long.toString(value, 10).toCharArray();
				}

				// leading sign indicator
				leadingSign(sb, neg);

				// the value
				localizedMagnitude(sb, va, this.f, adjustWidth(this.width, this.f, neg));

				// trailing sign indicator
				trailingSign(sb, neg);
			} else if (this.c == Conversion.OCTAL_INTEGER) {
				checkBadFlags(Flags.PARENTHESES, Flags.LEADING_SPACE, Flags.PLUS);
				String s = toOctalString(value);
				int len = (this.f.contains(Flags.ALTERNATE) ? s.length() + 1 : s.length());

				// apply ALTERNATE (radix indicator for octal) before ZERO_PAD
				if (this.f.contains(Flags.ALTERNATE)) {
					sb.append('0');
				}
				if (this.f.contains(Flags.ZERO_PAD)) {
					for (int i = 0; i < this.width - len; i++) {
						sb.append('0');
					}
				}
				sb.append(s);
			} else if (this.c == Conversion.HEXADECIMAL_INTEGER) {
				checkBadFlags(Flags.PARENTHESES, Flags.LEADING_SPACE, Flags.PLUS);
				String s = Long.toHexString(value);
				int len = (this.f.contains(Flags.ALTERNATE) ? s.length() + 2 : s.length());

				// apply ALTERNATE (radix indicator for hex) before ZERO_PAD
				if (this.f.contains(Flags.ALTERNATE)) {
					sb.append(this.f.contains(Flags.UPPERCASE) ? "0X" : "0x"); //$NON-NLS-1$ //$NON-NLS-2$
				}
				if (this.f.contains(Flags.ZERO_PAD)) {
					for (int i = 0; i < this.width - len; i++) {
						sb.append('0');
					}
				}
				if (this.f.contains(Flags.UPPERCASE)) {
					s = s.toUpperCase();
				}
				sb.append(s);
			}

			// justify based on width
			Appendable a = Formatter.this.a;
			assert a != null; // already ensured open in Formatter.format(...)
			a.append(justify(sb.toString()));
		}

		private String toOctalString(long l) {
			return toUnsignedStringLong(l, 3);
		}

		private String toUnsignedStringLong(long i, int shift) {
			char[] buffer = new char[64];
			int mask = (1 << shift) - 1;
			int ptrBuffer = 64;
			do {
				buffer[--ptrBuffer] = charValue(((int) i) & mask);
				i >>>= shift;
			} while (i != 0);
			return new String(buffer, ptrBuffer, 64 - ptrBuffer);
		}

		private char charValue(int digit) {
			// sequence of if statement is more efficient than a lookup swith
			if (digit >= 0 && digit <= 9) {
				return (char) (digit + '0');
			}
			if (digit >= 10 && digit <= 35) {
				return (char) (digit + ('a' - 10));
			}
			throw new IndexOutOfBoundsException();
		}

		// neg := val < 0
		private StringBuilder leadingSign(StringBuilder sb, boolean neg) {
			if (!neg) {
				if (this.f.contains(Flags.PLUS)) {
					sb.append('+');
				} else if (this.f.contains(Flags.LEADING_SPACE)) {
					sb.append(' ');
				}
			} else {
				if (this.f.contains(Flags.PARENTHESES)) {
					sb.append('(');
				} else {
					sb.append('-');
				}
			}
			return sb;
		}

		// neg := val < 0
		private StringBuilder trailingSign(StringBuilder sb, boolean neg) {
			if (neg && this.f.contains(Flags.PARENTHESES)) {
				sb.append(')');
			}
			return sb;
		}

		private void print(float value, Object arg) throws IOException {
			print((double) value, arg);
		}

		private void print(double value, Object arg) throws IOException {
			StringBuilder sb = new StringBuilder();
			boolean neg = Double.compare(value, 0.0) == -1;

			if (!Double.isNaN(value)) {
				double v = Math.abs(value);
				if (Double.compare(v, 0.0) == -1) {
					v = v * -1d; // Workaround for EDC bug.
				}

				// leading sign indicator
				leadingSign(sb, neg);

				// the value
				if (!Double.isInfinite(v)) {
					print(sb, v, this.f, this.c, this.precision, neg, arg);
				} else {
					sb.append(this.f.contains(Flags.UPPERCASE) ? "INFINITY" : "Infinity"); //$NON-NLS-1$ //$NON-NLS-2$
				}

				// trailing sign indicator
				trailingSign(sb, neg);
			} else {
				sb.append(this.f.contains(Flags.UPPERCASE) ? "NAN" : "NaN"); //$NON-NLS-1$ //$NON-NLS-2$
			}

			// justify based on width
			Appendable a = Formatter.this.a;
			assert a != null; // already ensured open in Formatter.format(...)
			a.append(justify(sb.toString()));
		}

		// !Double.isInfinite(value) && !Double.isNaN(value)
		private void print(StringBuilder sb, double value, Flags f, char c, int precision, boolean neg, Object arg)
				throws IOException {
			// if (c == Conversion.DECIMAL_FLOAT) {
			// Create a new FormattedFloatingDecimal with the desired
			// precision.
			int prec = (precision == -1 ? 6 : precision);

			FormattedFloatingDecimal fd = FormattedFloatingDecimal.valueOf(value, prec, arg);

			char[] mant = addZeros(fd.getMantissa(), prec);

			// If the precision is zero and the '#' flag is set, add the
			// requested decimal point.
			if (f.contains(Flags.ALTERNATE) && (prec == 0)) {
				mant = addDot(mant);
			}

			int newW = this.width;
			if (this.width != -1) {
				newW = adjustWidth(this.width, f, neg);
			}
			localizedMagnitude(sb, mant, f, newW);
			// }
		}

		// Add zeros to the requested precision.
		private char[] addZeros(char[] v, int prec) {
			// Look for the dot. If we don't find one, the we'll need to add
			// it before we add the zeros.
			int i;
			for (i = 0; i < v.length; i++) {
				if (v[i] == '.') {
					break;
				}
			}
			boolean needDot = false;
			if (i == v.length) {
				needDot = true;
			}

			// Determine existing precision.
			int outPrec = v.length - i - (needDot ? 0 : 1);
			assert (outPrec <= prec);
			if (outPrec == prec) {
				return v;
			}

			// Create new array with existing contents.
			char[] tmp = new char[v.length + prec - outPrec + (needDot ? 1 : 0)];
			System.arraycopy(v, 0, tmp, 0, v.length);

			// Add dot if previously determined to be necessary.
			int start = v.length;
			if (needDot) {
				tmp[v.length] = '.';
				start++;
			}

			// Add zeros.
			for (int j = start; j < tmp.length; j++) {
				tmp[j] = '0';
			}

			return tmp;
		}

		private int adjustWidth(int width, Flags f, boolean neg) {
			int newW = width;
			if (newW != -1 && neg && f.contains(Flags.PARENTHESES)) {
				newW--;
			}
			return newW;
		}

		// Add a '.' to th mantissa if required
		private char[] addDot(char[] mant) {
			char[] tmp = mant;
			tmp = new char[mant.length + 1];
			System.arraycopy(mant, 0, tmp, 0, mant.length);
			tmp[tmp.length - 1] = '.';
			return tmp;
		}

		// -- Methods to support throwing exceptions --

		private void failMismatch(Flags f, char c) {
			String fs = f.toString();
			throw new FormatFlagsConversionMismatchException(fs, c);
		}

		private void failConversion(char c, Object arg) {
			throw new IllegalFormatConversionException(c, arg.getClass());
		}

		private char getZero() {
			return '0';
		}

		private StringBuilder localizedMagnitude(@Nullable StringBuilder sb, char[] value, Flags f, int width) {
			if (sb == null) {
				sb = new StringBuilder();
			}
			int begin = sb.length();

			char zero = getZero();

			// determine localized grouping separator and size
			char grpSep = '\0';
			int grpSize = -1;
			char decSep = '\0';

			int len = value.length;
			int dot = len;
			for (int j = 0; j < len; j++) {
				if (value[j] == '.') {
					dot = j;
					break;
				}
			}

			if (dot < len) {
				decSep = '.';
			}

			// localize the digits inserting group separators as necessary
			for (int j = 0; j < len; j++) {
				if (j == dot) {
					sb.append(decSep);
					// no more group separators after the decimal separator
					grpSep = '\0';
					continue;
				}

				char c = value[j];
				sb.append((char) ((c - '0') + zero));
				if (grpSep != '\0' && j != dot - 1 && ((dot - j) % grpSize == 1)) {
					sb.append(grpSep);
				}
			}

			// apply zero padding
			len = sb.length();
			if (width != -1 && f.contains(Flags.ZERO_PAD)) {
				for (int k = 0; k < width - len; k++) {
					sb.insert(begin, zero);
				}
			}

			return sb;
		}
	}

	private static class Flags {
		private int flags;

		static final Flags NONE = new Flags(0); // ''

		// duplicate declarations from Formattable.java
		static final Flags LEFT_JUSTIFY = new Flags(1 << 0); // '-'
		static final Flags UPPERCASE = new Flags(1 << 1); // '^'
		static final Flags ALTERNATE = new Flags(1 << 2); // '#'

		// numerics
		static final Flags PLUS = new Flags(1 << 3); // '+'
		static final Flags LEADING_SPACE = new Flags(1 << 4); // ' '
		static final Flags ZERO_PAD = new Flags(1 << 5); // '0'
		static final Flags PARENTHESES = new Flags(1 << 7); // '('

		// indexing
		static final Flags PREVIOUS = new Flags(1 << 8); // '<'

		private Flags(int f) {
			this.flags = f;
		}

		public int valueOf() {
			return this.flags;
		}

		public boolean contains(Flags f) {
			return (this.flags & f.valueOf()) == f.valueOf();
		}

		public Flags dup() {
			return new Flags(this.flags);
		}

		private Flags add(Flags f) {
			this.flags |= f.valueOf();
			return this;
		}

		public Flags remove(Flags f) {
			this.flags &= ~f.valueOf();
			return this;
		}

		// parse those flags which may be provided by users
		private static Flags parse(char c) {
			switch (c) {
			case '-':
				return LEFT_JUSTIFY;
			case '#':
				return ALTERNATE;
			case '+':
				return PLUS;
			case ' ':
				return LEADING_SPACE;
			case '0':
				return ZERO_PAD;
			case '(':
				return PARENTHESES;
			case '<':
				return PREVIOUS;
			default:
				throw new UnknownFormatFlagsException(String.valueOf(c));
			}
		}

		@Override
		public String toString() {
			StringBuilder sb = new StringBuilder();
			if (contains(LEFT_JUSTIFY)) {
				sb.append('-');
			}
			if (contains(UPPERCASE)) {
				sb.append('^');
			}
			if (contains(ALTERNATE)) {
				sb.append('#');
			}
			if (contains(PLUS)) {
				sb.append('+');
			}
			if (contains(LEADING_SPACE)) {
				sb.append(' ');
			}
			if (contains(ZERO_PAD)) {
				sb.append('0');
			}
			if (contains(PARENTHESES)) {
				sb.append('(');
			}
			if (contains(PREVIOUS)) {
				sb.append('<');
			}
			return sb.toString();
		}
	}

	private static class Conversion {
		// Byte, Short, Integer, Long
		// (and associated primitives due to autoboxing)
		static final char DECIMAL_INTEGER = 'd';
		static final char OCTAL_INTEGER = 'o';
		static final char HEXADECIMAL_INTEGER = 'x';
		static final char HEXADECIMAL_INTEGER_UPPER = 'X';

		// Float, Double
		// (and associated primitives due to autoboxing)
		// static final char SCIENTIFIC = 'e';
		// static final char SCIENTIFIC_UPPER = 'E';
		// static final char GENERAL = 'g';
		// static final char GENERAL_UPPER = 'G';
		static final char DECIMAL_FLOAT = 'f';
		// static final char HEXADECIMAL_FLOAT = 'a';
		// static final char HEXADECIMAL_FLOAT_UPPER = 'A';

		// Character, Byte, Short, Integer
		// (and associated primitives due to autoboxing)
		static final char CHARACTER = 'c';
		static final char CHARACTER_UPPER = 'C';

		// if (arg.TYPE != boolean) return boolean
		// if (arg != null) return true; else return false;
		static final char BOOLEAN = 'b';
		static final char BOOLEAN_UPPER = 'B';
		// if (arg instanceof Formattable) arg.formatTo()
		// else arg.toString();
		static final char STRING = 's';
		static final char STRING_UPPER = 'S';
		// arg.hashCode()
		static final char HASHCODE = 'h';
		static final char HASHCODE_UPPER = 'H';

		static final char LINE_SEPARATOR = 'n';
		static final char PERCENT_SIGN = '%';

		static boolean isValid(char c) {
			return (isGeneral(c) || isInteger(c) || isFloat(c) || isText(c) || c == 't' || isCharacter(c));
		}

		// Returns true iff the Conversion is applicable to all objects.
		static boolean isGeneral(char c) {
			switch (c) {
			case BOOLEAN:
			case BOOLEAN_UPPER:
			case STRING:
			case STRING_UPPER:
			case HASHCODE:
			case HASHCODE_UPPER:
				return true;
			default:
				return false;
			}
		}

		// Returns true iff the Conversion is applicable to character.
		static boolean isCharacter(char c) {
			switch (c) {
			case CHARACTER:
			case CHARACTER_UPPER:
				return true;
			default:
				return false;
			}
		}

		// Returns true iff the Conversion is an integer type.
		static boolean isInteger(char c) {
			switch (c) {
			case DECIMAL_INTEGER:
			case OCTAL_INTEGER:
			case HEXADECIMAL_INTEGER:
			case HEXADECIMAL_INTEGER_UPPER:
				return true;
			default:
				return false;
			}
		}

		// Returns true iff the Conversion is a floating-point type.
		static boolean isFloat(char c) {
			switch (c) {
			// case SCIENTIFIC:
			// case SCIENTIFIC_UPPER:
			// case GENERAL:
			// case GENERAL_UPPER:
			case DECIMAL_FLOAT:
				// case HEXADECIMAL_FLOAT:
				// case HEXADECIMAL_FLOAT_UPPER:
				return true;
			default:
				return false;
			}
		}

		// Returns true iff the Conversion does not require an argument
		static boolean isText(char c) {
			switch (c) {
			case LINE_SEPARATOR:
			case PERCENT_SIGN:
				return true;
			default:
				return false;
			}
		}
	}

}
