/*
 * Java
 *
 * Copyright 2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.drawing.LLDWPainter.DrawingFlip;
import ej.microui.display.LLUIPainter.MicroUIGraphicsContext;
import ej.microui.display.LLUIPainter.MicroUIImage;
import ej.microui.display.LLUIPainter.MicroUIImageFormat;

/**
 * This interface provides all the image drawing methods.
 * <p>
 * This interface can be fully implemented by a dedicated platform class in platform front panel project. This allows to
 * draw and decode images with a custom format.
 */
public interface UIImageDrawing {

	/**
	 * Gets the supported custom image format.
	 *
	 * @return the supported custom image format
	 */
	MicroUIImageFormat handledFormat();

	/**
	 * Draws a region of an image. The region of the image to draw is given relative to the image (origin at the
	 * upper-left corner) as a rectangle.
	 * <p>
	 * If the specified source region exceeds the image bounds, the copied region is limited to the image boundary. If
	 * the copied region goes out of the bounds of the graphics context area, pixels out of the range will not be drawn.
	 * <p>
	 * A global opacity value is given. When this value is 0xff (255, opaque), that means the image is drawn on the
	 * graphics context without managing an extra opacity. Only the image transparent pixels must have been merged with
	 * destination. All image opaque pixels override destination.
	 * <p>
	 * When this value is a value between 0 and 0xff, that means each pixel of the image must be merged with destination
	 * in addition with the image transparent pixels. An image opaque pixel becomes transparent (its opacity is the
	 * given alpha) and the opacity of an image transparent pixel becomes (alpha * alpha(pixel)) / 255.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param regionY
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void draw(MicroUIGraphicsContext gc, MicroUIImage img, int regionX, int regionY, int width, int height,
			int x, int y, int alpha) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an image applying a flip (0, 90, 180 or 270 degrees with or without mirror).
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to draw.
	 * @param regionY
	 *            the x coordinate of the upper-left corner of the region to draw.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param flip
	 *            the flip to apply.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawFlipped(MicroUIGraphicsContext gc, MicroUIImage img, int regionX, int regionY, int width,
			int height, int x, int y, DrawingFlip flip, int alpha) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an image applying a free rotation (0 to 360 degrees).
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative where the graphical object will be drawn.
	 * <p>
	 * This method uses the nearest neighbor algorithm to render the content. This algorithm is faster than bilinear
	 * algorithm.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param rotationX
	 *            the x coordinate of the rotation center.
	 * @param rotationY
	 *            the y coordinate of the rotation center.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawRotatedNearestNeighbor(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, int rotationX,
			int rotationY, float angle, int alpha) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an image applying a free rotation (0 to 360 degrees).
	 * <p>
	 * The rotation is specified by the center and the angle. The reference point is the graphical object top-left
	 * corner. The rotation point is relative where the graphical object will be drawn.
	 * <p>
	 * This method uses the bilinear algorithm to render the content. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param rotationX
	 *            the x coordinate of the rotation center.
	 * @param rotationY
	 *            the y coordinate of the rotation center.
	 * @param angle
	 *            the rotation angle.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawRotatedBilinear(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, int rotationX,
			int rotationY, float angle, int alpha) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an image applying a scaling.
	 * <p>
	 * This method uses the nearest neighbor algorithm to render the content. This algorithm is faster than bilinear
	 * algorithm.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param factorX
	 *            scaling X factor.
	 * @param factorY
	 *            scaling Y factor.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawScaledNearestNeighbor(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, float factorX,
			float factorY, int alpha) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Draws an image applying a scaling.
	 * <p>
	 * This method uses the bilinear algorithm to render the content. This algorithm performs better rendering than
	 * nearest neighbor algorithm but it is slower to apply.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param x
	 *            the x coordinate of the image reference anchor top-left point.
	 * @param y
	 *            the y coordinate of the image reference anchor top-left point.
	 * @param factorX
	 *            scaling X factor.
	 * @param factorY
	 *            scaling Y factor.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	default void drawScaledBilinear(MicroUIGraphicsContext gc, MicroUIImage img, int x, int y, float factorX,
			float factorY, int alpha) {
		gc.reportError(LLUIPainter.DRAWING_LOG_NOT_IMPLEMENTED);
	}

	/**
	 * Decodes the image generated by the Image Generator.
	 * <p>
	 * The image buffer encoding is shared between the Image Generator, the embedded VEE Port and this implementation.
	 * On the embedded VEE Port, the data should be used as is (no runtime decoding step) and should be optimized to be
	 * as fast as possible. On the simulator, the data has to be decoded at runtime.
	 * <p>
	 * The data is encoded in the given byte array at given offset. Its endianness depends on the embedded VEE Port
	 * endianness.
	 * <p>
	 * The default implementation returns <code>null</code> that means that the implementation is not able to use the
	 * encoded images.
	 *
	 * @param data
	 *            the image encoded data.
	 * @param offset
	 *            the index of the first data.
	 * @param width
	 *            the image width in pixels.
	 * @param height
	 *            the image height in pixels.
	 * @param isBigEndian
	 *            <code>true</code> when the embedded VEE Port endianness is big endian, <code>false</code> otherwise.
	 *
	 * @return an object that represents the image.
	 */
	default Object decode(byte[] data, int offset, int width, int height, boolean isBigEndian) {
		// Stub
		return null;
	}

}
