/*
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.outline;

import ej.annotation.Nullable;
import ej.bon.XMath;
import ej.microui.display.GraphicsContext;
import ej.mwt.util.Outlineable;
import ej.mwt.util.Size;

/**
 * Outline that have the same thickness for all edges.
 * <p>
 * The thickness is stored as a <code>char</code> for heap optimization and therefore cannot exceed <code>65535</code>.
 */
public class UniformOutline implements Outline {

	private final char thickness;

	/**
	 * Creates a uniform outline specifying its thickness.
	 * <p>
	 * The given thickness is clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param thickness
	 *            the thickness.
	 */
	public UniformOutline(int thickness) {
		this.thickness = (char) XMath.limit(thickness, 0, Character.MAX_VALUE);
	}

	/**
	 * Gets the thickness.
	 *
	 * @return the thickness.
	 */
	public int getThickness() {
		return this.thickness;
	}

	@Override
	public void apply(Outlineable outlineable) {
		int thickness = this.thickness;
		outlineable.removeOutline(thickness, thickness, thickness, thickness);
	}

	@Override
	public void apply(GraphicsContext g, Size size) {
		int thickness = this.thickness;
		size.removeOutline(thickness, thickness, thickness, thickness);

		g.translate(thickness, thickness);
		g.intersectClip(0, 0, size.getWidth(), size.getHeight());
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && getClass() == obj.getClass()) {
			UniformOutline outline = (UniformOutline) obj;
			return this.thickness == outline.thickness;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return 19 * this.thickness;
	}

}
