/*
 * Copyright (c) 2001, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2020 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * NOTE: This class lives in the package sun.net.www.protocol.https.
 * There is a copy in com.sun.net.ssl.internal.www.protocol.https for JSSE
 * 1.0.2 compatibility. It is 100% identical except the package and extends
 * lines. Any changes should be made to be class in sun.net.* and then copied
 * to com.sun.net.*.
 */

// For both copies of the file, uncomment one line and comment the other
package sun.net.www.protocol.https;
// package com.sun.net.ssl.internal.www.protocol.https;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ProtocolException;
import java.net.URL;
import java.security.Permission;
import java.util.List;
import java.util.Map;

import ej.annotation.Nullable;

/**
 * A class to represent an HTTP connection to a remote object.
 *
 * Ideally, this class should subclass and inherit the http handler implementation, but it can't do so because that
 * class have the wrong Java Type. Thus it uses the delegate (aka, the Adapter/Wrapper design pattern) to reuse code
 * from the http handler.
 *
 * Since it would use a delegate to access sun.net.www.protocol.http.HttpURLConnection functionalities, it needs to
 * implement all public methods in it's super class and all the way to Object.
 *
 */

// For both copies of the file, uncomment one line and comment the
// other. The differences between the two copies are introduced for
// plugin, and it is marked as such.
public class HttpsURLConnectionImpl extends javax.net.ssl.HttpsURLConnection {
	// public class HttpsURLConnectionOldImpl
	// extends com.sun.net.ssl.HttpsURLConnection {

	// NOTE: made protected for plugin so that subclass can set it.
	protected DelegateHttpsURLConnection delegate;

	// For both copies of the file, uncomment one line and comment the other
	HttpsURLConnectionImpl(URL u, Handler handler) throws IOException {
		this(u);
		this.delegate = new DelegateHttpsURLConnection(this.url, handler, this);
	}

	// NOTE: introduced for plugin
	// subclass needs to overwrite this to set delegate to
	// the appropriate delegatee
	protected HttpsURLConnectionImpl(URL u) throws IOException {
		super(u);
	}

	/**
	 * Create a new HttpClient object, bypassing the cache of HTTP client objects/connections.
	 *
	 * @param url
	 *            the URL being accessed
	 */
	protected void setNewClient(URL url) throws IOException {
		this.delegate.setNewClient(url);
	}

	/**
	 * Implements the HTTP protocol handler's "connect" method, establishing an SSL connection to the server as
	 * necessary.
	 */
	@Override
	public void connect() throws IOException {
		this.delegate.connect();
	}

	/**
	 * Used by subclass to access "connected" variable. Since we are delegating the actual implementation to "delegate",
	 * we need to delegate the access of "connected" as well.
	 */
	protected boolean isConnected() {
		return this.delegate.isConnected();
	}

	/**
	 * Used by subclass to access "connected" variable. Since we are delegating the actual implementation to "delegate",
	 * we need to delegate the access of "connected" as well.
	 */
	protected void setConnected(boolean conn) {
		this.delegate.setConnected(conn);
	}

	/*
	 * Allowable input/output sequences: [interpreted as POST/PUT] - get output, [write output,] get input, [read input]
	 * - get output, [write output] [interpreted as GET] - get input, [read input] Disallowed: - get input, [read
	 * input,] get output, [write output]
	 */

	@Override
	public synchronized OutputStream getOutputStream() throws IOException {
		return this.delegate.getOutputStream();
	}

	@Override
	public synchronized InputStream getInputStream() throws IOException {
		return this.delegate.getInputStream();
	}

	@Override
	@Nullable
	public InputStream getErrorStream() {
		return this.delegate.getErrorStream();
	}

	/**
	 * Disconnect from the server.
	 */
	@Override
	public void disconnect() {
		this.delegate.disconnect();
	}

	@Override
	public boolean usingProxy() {
		return this.delegate.usingProxy();
	}

	/**
	 * Returns an unmodifiable Map of the header fields. The Map keys are Strings that represent the response-header
	 * field names. Each Map value is an unmodifiable List of Strings that represents the corresponding field values.
	 *
	 * @return a Map of header fields
	 * @since 1.4
	 */
	@Override
	public Map<String, List<String>> getHeaderFields() {
		return this.delegate.getHeaderFields();
	}

	/**
	 * Gets a header field by name. Returns null if not known.
	 *
	 * @param name
	 *            the name of the header field
	 */
	@Override
	@Nullable
	public String getHeaderField(String name) {
		return this.delegate.getHeaderField(name);
	}

	/**
	 * Gets a header field by index. Returns null if not known.
	 *
	 * @param n
	 *            the index of the header field
	 */
	@Override
	@Nullable
	public String getHeaderField(int n) {
		return this.delegate.getHeaderField(n);
	}

	/**
	 * Gets a header field by index. Returns null if not known.
	 *
	 * @param n
	 *            the index of the header field
	 */
	@Override
	@Nullable
	public String getHeaderFieldKey(int n) {
		return this.delegate.getHeaderFieldKey(n);
	}

	/**
	 * Sets request property. If a property with the key already exists, overwrite its value with the new value.
	 *
	 * @param value
	 *            the value to be set
	 */
	@Override
	public void setRequestProperty(String key, String value) {
		this.delegate.setRequestProperty(key, value);
	}

	/**
	 * Adds a general request property specified by a key-value pair. This method will not overwrite existing values
	 * associated with the same key.
	 *
	 * @param key
	 *            the keyword by which the request is known (e.g., "<code>accept</code>").
	 * @param value
	 *            the value associated with it.
	 * @see #getRequestProperties()
	 * @since 1.4
	 */
	@Override
	public void addRequestProperty(String key, String value) {
		this.delegate.addRequestProperty(key, value);
	}

	/**
	 * Overwrite super class method
	 */
	@Override
	public int getResponseCode() throws IOException {
		return this.delegate.getResponseCode();
	}

	@Override
	@Nullable
	public String getRequestProperty(String key) {
		return this.delegate.getRequestProperty(key);
	}

	/**
	 * Returns an unmodifiable Map of general request properties for this connection. The Map keys are Strings that
	 * represent the request-header field names. Each Map value is a unmodifiable List of Strings that represents the
	 * corresponding field values.
	 *
	 * @return a Map of the general request properties for this connection.
	 * @throws IllegalStateException
	 *             if already connected
	 * @since 1.4
	 */
	@Override
	public Map<String, List<String>> getRequestProperties() {
		return this.delegate.getRequestProperties();
	}

	/*
	 * We support JDK 1.2.x so we can't count on these from JDK 1.3. We override and supply our own version.
	 */
	@Override
	public void setInstanceFollowRedirects(boolean shouldFollow) {
		this.delegate.setInstanceFollowRedirects(shouldFollow);
	}

	@Override
	public boolean getInstanceFollowRedirects() {
		return this.delegate.getInstanceFollowRedirects();
	}

	@Override
	public void setRequestMethod(String method) throws ProtocolException {
		this.delegate.setRequestMethod(method);
	}

	@Override
	public String getRequestMethod() {
		return this.delegate.getRequestMethod();
	}

	@Override
	@Nullable
	public String getResponseMessage() throws IOException {
		return this.delegate.getResponseMessage();
	}

	@Override
	public Permission getPermission() throws IOException {
		return this.delegate.getPermission();
	}

	@Override
	public URL getURL() {
		return this.delegate.getURL();
	}

	@Override
	public int getContentLength() {
		return this.delegate.getContentLength();
	}

	@Override
	public long getContentLengthLong() {
		return this.delegate.getContentLengthLong();
	}

	@Override
	@Nullable
	public String getContentType() {
		return this.delegate.getContentType();
	}

	@Override
	@Nullable
	public String getContentEncoding() {
		return this.delegate.getContentEncoding();
	}

	@Override
	public int getHeaderFieldInt(String name, int Default) {
		return this.delegate.getHeaderFieldInt(name, Default);
	}

	@Override
	public long getHeaderFieldLong(String name, long Default) {
		return this.delegate.getHeaderFieldLong(name, Default);
	}

	@Override
	public Object getContent() throws IOException {
		return this.delegate.getContent();
	}

	@Override
	@SuppressWarnings("rawtypes")
	@Nullable
	public Object getContent(Class[] classes) throws IOException {
		return this.delegate.getContent(classes);
	}

	@Override
	public String toString() {
		return this.delegate.toString();
	}

	@Override
	public void setDoInput(boolean doinput) {
		this.delegate.setDoInput(doinput);
	}

	@Override
	public boolean getDoInput() {
		return this.delegate.getDoInput();
	}

	@Override
	public void setDoOutput(boolean dooutput) {
		this.delegate.setDoOutput(dooutput);
	}

	@Override
	public boolean getDoOutput() {
		return this.delegate.getDoOutput();
	}

	@Override
	public void setAllowUserInteraction(boolean allowuserinteraction) {
		this.delegate.setAllowUserInteraction(allowuserinteraction);
	}

	@Override
	public boolean getAllowUserInteraction() {
		return this.delegate.getAllowUserInteraction();
	}

	@Override
	public void setUseCaches(boolean usecaches) {
		this.delegate.setUseCaches(usecaches);
	}

	@Override
	public boolean getUseCaches() {
		return this.delegate.getUseCaches();
	}

	@Override
	public boolean getDefaultUseCaches() {
		return this.delegate.getDefaultUseCaches();
	}

	@Override
	public void setDefaultUseCaches(boolean defaultusecaches) {
		this.delegate.setDefaultUseCaches(defaultusecaches);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		return this.delegate.equals(obj);
	}

	@Override
	public int hashCode() {
		return this.delegate.hashCode();
	}

	@Override
	public void setConnectTimeout(int timeout) {
		this.delegate.setConnectTimeout(timeout);
	}

	@Override
	public int getConnectTimeout() {
		return this.delegate.getConnectTimeout();
	}

	@Override
	public void setReadTimeout(int timeout) {
		this.delegate.setReadTimeout(timeout);
	}

	@Override
	public int getReadTimeout() {
		return this.delegate.getReadTimeout();
	}

	@Override
	public void setFixedLengthStreamingMode(int contentLength) {
		this.delegate.setFixedLengthStreamingMode(contentLength);
	}

	@Override
	public void setFixedLengthStreamingMode(long contentLength) {
		this.delegate.setFixedLengthStreamingMode(contentLength);
	}

	@Override
	public void setChunkedStreamingMode(int chunklen) {
		this.delegate.setChunkedStreamingMode(chunklen);
	}
}
