/*
 * Java
 *
 * Copyright 2009-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.MouseListener;
import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.fp.util.WidgetWithListener;
import ej.microui.event.EventButton;

/**
 * Basic widget to simulate a button with two states: pressed and released.
 * <p>
 * Notes:
 * <ul>
 * <li>This widget requires a listener: an implementation of {@link ButtonListener} to send the button events,</li>
 * <li>The widget label must be an integer. It allows to create a MicroUI event using this button ID.</li>
 * </ul>
 */
@WidgetDescription(attributes = { @WidgetAttribute(name = "label"), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "skin"), @WidgetAttribute(name = "pushedSkin"),
		@WidgetAttribute(name = "filter", isOptional = true),
		@WidgetAttribute(name = "listenerClass", isOptional = true) })
public class Button extends WidgetWithListener implements MouseListener {

	/**
	 * Interface that handle button events.
	 */
	public static interface ButtonListener {

		/**
		 * The specified button has been pressed.
		 *
		 * @param widget
		 *            the button that has been pressed.
		 */
		void press(Button widget);

		/**
		 * The specified button has been released.
		 *
		 * @param widget
		 *            the button that has been released.
		 */
		void release(Button widget);
	}

	/**
	 * Default implementation of {@link ButtonListener}.
	 * <p>
	 * This implementation sends some MicroUI Buttons events using the button ID. It targets the common MicroUI Button
	 * generator identified by the tag {@link EventButton#COMMON_MICROUI_GENERATOR_TAG}.
	 * <p>
	 * This tag can be changed overriding the method {@link #getMicroUIGeneratorTag()}.
	 */
	public static class ButtonListenerToButtonEvents implements ButtonListener {

		@Override
		public void press(Button widget) {
			EventButton.sendPressedEvent(getMicroUIGeneratorTag(), widget.getID());
		}

		@Override
		public void release(Button widget) {
			EventButton.sendReleasedEvent(getMicroUIGeneratorTag(), widget.getID());
		}

		/**
		 * Gets the MicroUI Buttons events generator tag. This generator has to match the generator set during the
		 * MicroEJ platform build in <code>microui/microui.xml</code>
		 *
		 * @return a MicroUI Buttons events generator tag
		 */
		protected String getMicroUIGeneratorTag() {
			return EventButton.COMMON_MICROUI_GENERATOR_TAG;
		}
	}

	private Image pushedSkin;
	private ButtonListener listener;
	private int id;

	@Override
	public void setLabel(String label) {
		super.setLabel(label);

		// label must be integer
		this.id = Integer.parseInt(getLabel());
	}

	/**
	 * Sets the image which shows the pressed button.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param pushedSkin
	 *            the button pressed image.
	 */
	public void setPushedSkin(Image pushedSkin) {
		this.pushedSkin = pushedSkin;
	}

	/**
	 * Defines the user class which has to implement {@link ButtonListener}.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param listenerClassName
	 *            user listener class name.
	 */
	public void setListenerClass(String listenerClassName) {
		setListenerClass(ButtonListener.class, listenerClassName);
	}

	@Override
	public void start() {
		super.start();

		// retrieve the listener
		this.listener = newListener(ButtonListener.class);
	}

	@Override
	public void dispose() {
		FrontPanel.getFrontPanel().disposeIfNotNull(this.pushedSkin);
		this.pushedSkin = null;
		super.dispose();
	}

	@Override
	protected Object newDefaultListener() {
		return new ButtonListenerToButtonEvents();
	}

	@Override
	public void mousePressed(int x, int y, MouseButton button) {
		if (button == MouseButton.FIRST_BUTTON) {
			press();
		}
	}

	@Override
	public void mouseReleased(int x, int y, MouseButton button) {
		if (button == MouseButton.FIRST_BUTTON) {
			release();
		}
	}

	/**
	 * Default implementation of press action: update the skin and ask to the listener to manage the event.
	 */
	protected void press() {
		// set pushed skin as current
		setCurrentSkin(this.pushedSkin);
		this.listener.press(this);
	}

	/**
	 * Default implementation of release action: update the skin and ask to the listener to manage the event.
	 */
	protected void release() {
		// set default skin as current
		setCurrentSkin(getSkin());
		this.listener.release(this);
	}

	/**
	 * Gets the widget listener.
	 *
	 * @return the listener which manages the events.
	 */
	protected ButtonListener getListener() {
		return this.listener;
	}

	/**
	 * Gets the skin which shows the button as pressed.
	 *
	 * @return the pushed skin
	 */
	public Image getPushedSkin() {
		return this.pushedSkin;
	}

	/**
	 * Gets the button identifier (equals to its label).
	 *
	 * @return the button identifier.
	 */
	public int getID() {
		return this.id;
	}
}
