/*
 * Copyright (c) 1996, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2018, IS2T. EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.security;

/**
 * This class provides a cryptographically strong random number generator (RNG).
 *
 * <p>
 * A cryptographically strong random number minimally complies with the statistical random number generator tests
 * specified in <a href="http://csrc.nist.gov/cryptval/140-2.htm"> <i>FIPS 140-2, Security Requirements for
 * Cryptographic Modules</i></a>, section 4.9.1. Additionally, SecureRandom must produce non-deterministic output.
 * Therefore any seed material passed to a SecureRandom object must be unpredictable, and all SecureRandom output
 * sequences must be cryptographically strong, as described in <a href="http://www.ietf.org/rfc/rfc1750.txt"> <i>RFC
 * 1750: Randomness Recommendations for Security</i></a>.
 *
 * <p>
 * A caller obtains a SecureRandom instance via the no-argument constructor:
 *
 * <pre>
 * SecureRandom random = new SecureRandom();
 * </pre>
 *
 * <p>
 * Many SecureRandom implementations are in the form of a pseudo-random number generator (PRNG), which means they use a
 * deterministic algorithm to produce a pseudo-random sequence from a true random seed. Other implementations may
 * produce true random numbers, and yet others may use a combination of both techniques.
 *
 * <p>
 * Typical callers of SecureRandom invoke the following methods to retrieve random bytes:
 *
 * <pre>
 * SecureRandom random = new SecureRandom();
 * byte bytes[] = new byte[20];
 * random.nextBytes(bytes);
 * </pre>
 *
 * <p>
 * Callers may also invoke the {@code generateSeed} method to generate a given number of seed bytes (to seed other
 * random number generators, for example):
 *
 * <pre>
 * byte seed[] = random.generateSeed(20);
 * </pre>
 *
 * Note: Depending on the implementation, the {@code generateSeed} and {@code nextBytes} methods may block as entropy is
 * being gathered, for example, if they need to read from /dev/random on various Unix-like operating systems.
 *
 * @see java.util.Random
 *
 * @author Benjamin Renaud
 * @author Josh Bloch
 */

public class SecureRandom extends java.util.Random {

	/**
	 * Constructs a secure random number generator (RNG) implementing the default random number algorithm.
	 *
	 * <p>
	 * The returned SecureRandom object has not been seeded. To seed the returned object, call the {@code setSeed}
	 * method. If {@code setSeed} is not called, the first call to {@code nextBytes} will force the SecureRandom object
	 * to seed itself. This self-seeding will not occur if {@code setSeed} was previously called.
	 */
	public SecureRandom() {
		throw new RuntimeException();
	}

	/**
	 * Constructs a secure random number generator (RNG) implementing the default random number algorithm. The
	 * SecureRandom instance is seeded with the specified seed bytes.
	 *
	 * @param seed
	 *            the seed.
	 */
	public SecureRandom(byte seed[]) {
		throw new RuntimeException();
	}

	/**
	 * Reseeds this random object. The given seed supplements, rather than replaces, the existing seed. Thus, repeated
	 * calls are guaranteed never to reduce randomness.
	 *
	 * @param seed
	 *            the seed.
	 */
	synchronized public void setSeed(byte[] seed) {
		throw new RuntimeException();
	}

	/**
	 * Reseeds this random object, using the eight bytes contained in the given {@code long seed}. The given seed
	 * supplements, rather than replaces, the existing seed. Thus, repeated calls are guaranteed never to reduce
	 * randomness.
	 *
	 * <p>
	 * This method is defined for compatibility with {@code java.util.Random}.
	 *
	 * @param seed
	 *            the seed.
	 */
	@Override
	public void setSeed(long seed) {
		throw new RuntimeException();
	}

	/**
	 * Generates a user-specified number of random bytes.
	 *
	 * <p>
	 * If a call to {@code setSeed} had not occurred previously, the first call to this method forces this SecureRandom
	 * object to seed itself. This self-seeding will not occur if {@code setSeed} was previously called.
	 *
	 * @param bytes
	 *            the array to be filled in with random bytes.
	 */
	@Override
	public void nextBytes(byte[] bytes) {
		throw new RuntimeException();
	}

	/**
	 * Generates an integer containing the user-specified number of pseudo-random bits (right justified, with leading
	 * zeros). This method overrides a {@code java.util.Random} method, and serves to provide a source of random bits to
	 * all of the methods inherited from that class (for example, {@code nextInt}, {@code nextLong}, and
	 * {@code nextFloat}).
	 *
	 * @param numBits
	 *            number of pseudo-random bits to be generated, where {@code 0 <= numBits <= 32}.
	 *
	 * @return an {@code int} containing the user-specified number of pseudo-random bits (right justified, with leading
	 *         zeros).
	 */
	@Override
	final protected int next(int numBits) {
		throw new RuntimeException();
	}

	/**
	 * Returns the given number of seed bytes, computed using the seed generation algorithm that this class uses to seed
	 * itself. This call may be used to seed other random number generators.
	 *
	 * @param numBytes
	 *            the number of seed bytes to generate.
	 *
	 * @return the seed bytes.
	 */
	public byte[] generateSeed(int numBytes) {
		throw new RuntimeException();
	}

	// Declare serialVersionUID to be compatible with JDK1.1
	static final long serialVersionUID = 4940670005562187L;

}
