/*
 * Java
 *
 *	Copyright 2025 MicroEJ Corp. All rights reserved.
 *	Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.module;

import ej.annotation.Nullable;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;

import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Module manager.
 */
public class SandboxedModuleManager implements SandboxedModuleHelper, FeatureStateListener {
	private static final Logger LOGGER = Logger.getLogger("ModuleManager");
	private final ArrayList<SandboxedModule> modules;

	public SandboxedModuleManager() {
		this.modules = new ArrayList<>();
		this.modules.add(new KernelModule());

		// -----------------------
		// Actions on Apps State Changes
		// Registers a listener to monitor application state changes: Start, Stop, Install, Uninstall.
		// -----------------------
		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.info("Registering Application State Change Listener");
		}
		Kernel.addFeatureStateListener(this);
	}

	@Override
	@Nullable
	public SandboxedModule getModule(String moduleName) {
		synchronized (this.modules) {
			for (SandboxedModule module : this.modules) {
				if (module.getIdentifier().equals(moduleName)) {
					return module;
				}
			}
		}
		return null;
	}

	@Override
	public SandboxedModule getCurrentModule() {
		Module module = Kernel.getContextOwner();

		Kernel.enter();

		return getModule(module.getName());
	}

	@Override
	public void stateChanged(Feature app, @Nullable State previousState) {

		switch (app.getState()) {
		case STARTED:
			onStarted(app);
			break;

		case STOPPED:
			onStopped(app);
			break;

		case INSTALLED:
			if (previousState == null) {
				onInstalled(app);
			} else if (State.STOPPED.equals(previousState)) {
				onStopCompleted(app);
			}
			break;

		case UNINSTALLED:
			onUninstalled(app);
			break;

		default:
			break;
		}
	}

	private void onInstalled(Feature app) {
		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.info("New application installation detected: " + app.getName());
		}

		ApplicationModule appModule = new ApplicationModule(app);

		this.add(appModule);

		try {
			appModule.applyPolicy();
		} catch (Exception e) {
			if (LOGGER.isLoggable(Level.SEVERE)) {
				LOGGER.log(Level.SEVERE, "Error while applying the policy file", e);
			}

			/* TODO: remove application when an error occurs while parsing the policy file. */
		}
	}

	private void onStarted(Feature app) {
		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.info("Application running: " + app.getName());
		}

		ApplicationModule appModule = (ApplicationModule) this.getModule(app.getName());
		if (appModule != null) {
			appModule.applyThreadPriority();
		}
	}

	private void onStopped(Feature app) {
		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.info("Application stopped: " + app.getName());
		}

		ApplicationModule appModule = (ApplicationModule) this.getModule(app.getName());
		if (appModule != null) {
			appModule.getFileSystemResourceController().reset();
			appModule.getNetworkResourceController().reset();

			this.remove(appModule);
		}
	}

	private void onStopCompleted(Feature app) {
		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.info("Application completely stopped, and can be uninstalled: " + app.getName());
		}
	}

	private void onUninstalled(Feature app) {
		if (LOGGER.isLoggable(Level.INFO)) {
			LOGGER.info("Application uninstalled: " + app.getName());
		}
	}

	private void add(ApplicationModule app) {
		synchronized (this.modules) {
			this.modules.add(app);
		}
	}

	private void remove(ApplicationModule app) {
		synchronized (this.modules) {
			this.modules.remove(app);
		}
	}
}
