/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.net;

import java.net.InetAddress;
import java.net.UnknownHostException;

/**
 * Subnet representation with its network address and mask in bits.
 *
 * E.g: For the subnet 28.10.0.0/16, 28.10.0.0 represents the network address and 16 is the mask in bits.
 */
public class Subnet {

	private final InetAddress networkAddress;
	private final int maskBits;

	/**
	 * @param networkAddress
	 * 		the network address of the subnet.
	 * @param mask
	 * 		the mask of the subnet (in bits).
	 */
	public Subnet(InetAddress networkAddress, int mask) {
		super();
		this.networkAddress = networkAddress;
		this.maskBits = mask;
	}

	/**
	 * Gets the network address of the subnet.
	 *
	 * @return the network address of the subnet.
	 */
	public InetAddress getNetworkAddress() {
		return this.networkAddress;
	}

	/**
	 * Gets the mask of the subnet (in bits).
	 *
	 * @return the mask of the subnet (in bits).
	 */
	public int getMask() {
		return this.maskBits;
	}

	/**
	 * Checks if the given ip address matches with the subnet.
	 *
	 * @param ipAddress
	 * 		the ip address.
	 * @return true on success; false otherwise.
	 */
	public boolean matches(InetAddress ipAddress) {
		InetAddress networkAddress = getNetworkAddress();
		int maskBits = getMask();
		// Check if the subnet mask is specified.
		if (maskBits < 0) {
			// No mask specified.
			// We will match only if the both addresses are the same.
			return networkAddress.equals(ipAddress);
		}

		byte[] networkAddressBytes = networkAddress.getAddress();
		byte[] ipAddressBytes = ipAddress.getAddress();

		// Check if the ip address and the network address are in the same address family (ipv4 ou ipv6).
		if (!networkAddress.getClass().equals(ipAddress.getClass())) {
			// No same ip address family.
			return false;
		}

		int maskFullBytes = maskBits / 8;

		// Check the full bytes part of the mask (the mask bytes that have all bits "on").
		for (int i = 0; i < maskFullBytes; i++) {
			if (ipAddressBytes[i] != networkAddressBytes[i]) {
				return false;
			}
		}
		// Compute the last byte of the mask to check.
		int lastMaskBits = maskBits % 8;
		if (lastMaskBits != 0) {
			byte lastMaskByte = (byte) (0xFF << (8 - lastMaskBits));
			// Apply the last mask byte to its corresponding byte part of the IPs addresses.
			return (networkAddressBytes[maskFullBytes] & lastMaskByte) == (ipAddressBytes[maskFullBytes]
					& lastMaskByte);
		}
		return true;
	}

	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof Subnet)) {
			return false;
		}
		Subnet other = (Subnet) obj;
		return this.networkAddress.equals(other.getNetworkAddress()) && this.maskBits == other.maskBits;
	}

	@Override
	public int hashCode() {
		return toString().hashCode();
	}

	@Override
	public String toString() {
		return this.networkAddress.toString() + "/" + this.maskBits; // $NON-NLS-1$
	}

	/**
	 * Creates an instance of {@code Subnet} from the given subnet string which is in CIDR notation (e.g:
	 * 192.168.1.0/24).
	 *
	 * @param cidrSubnet
	 * 		the subnet in CIDR notation.
	 * @return {@code Subnet} instance from the given CIDR notation.
	 * @throws UnknownHostException
	 * 		if the subnet IP address cannot be resolved.
	 */
	public static Subnet fromCidrNotation(String cidrSubnet) throws UnknownHostException {
		InetAddress networkID = null;
		int maskBits;
		int maskIndex = cidrSubnet.lastIndexOf("/"); // $NON-NLS-1$
		if (maskIndex < 0) {
			// No mask specified.
			maskBits = -1;
			networkID = InetAddress.getByName(cidrSubnet);
		} else {
			networkID = InetAddress.getByName(cidrSubnet.substring(0, maskIndex));
			maskBits = Byte.parseByte(cidrSubnet.substring(maskIndex + 1));
		}
		return new Subnet(networkID, maskBits);
	}
}
