/*
 * Java
 *
 * Copyright 2009-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp.widget;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.Widget;
import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.microui.led.LLUILedImpl;

/**
 * This widget is an implementation of MicroUI {@link LLUILedImpl} interface. An implementation of this interface is
 * required to be able to use a MicroUI Led in a MicroEJ application.
 */
@WidgetDescription(attributes = { @WidgetAttribute(name = "label"), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "ledOff"), @WidgetAttribute(name = "ledOn"),
		@WidgetAttribute(name = "overlay", isOptional = true) })
public class LED extends Widget implements LLUILedImpl {

	private int intensity;

	// defaultSkin is ledOff
	private Image ledOn;

	/**
	 * Sets the image which represents the LED <i>off</i>.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param ledOff
	 *            the LED <i>off</i> image.
	 */
	public void setLedOff(Image ledOff) {
		setSkin(ledOff);
	}

	/**
	 * Sets the image which represents the LED <i>on</i>.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param ledOn
	 *            the LED <i>on</i> image.
	 */
	public void setLedOn(Image ledOn) {
		this.ledOn = ledOn;
	}

	@Override
	public void finalizeConfiguration() {
		super.finalizeConfiguration();
		this.intensity = MIN_INTENSITY; // default value
	}

	@Override
	public void showYourself(boolean appearSwitchedOn) {
		setIntensity(appearSwitchedOn ? MAX_INTENSITY : MIN_INTENSITY);
	}

	@Override
	public void dispose() {
		FrontPanel.getFrontPanel().disposeIfNotNull(this.ledOn);
		this.ledOn = null;
		super.dispose();
	}

	@Override
	public void setIntensity(int intensity) {
		if (intensity == 0) {
			setCurrentSkin(getSkin()); // led is off
			this.intensity = MIN_INTENSITY;
		} else {
			setCurrentSkin(this.ledOn); // led is on
			this.intensity = MAX_INTENSITY;
		}
	}

	@Override
	public int getIntensity() {
		return this.intensity;
	}
}
