/*
 * Java
 *
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.image.pathdata;

import java.nio.ByteBuffer;

/**
 * Decodes a path whose data is stored in two byte arrays: the commands and the commands' attributes. A command is
 * encoded on one byte and the command's attributes are encoded in float.
 */
public class TwoArraysPathData implements PathData {

	/**
	 * The number of bytes used to encode a command.
	 */
	public static final int COMMAND_SIZE = 1; // sizeof(byte)

	/**
	 * The number of bytes used to encode a command's attribute.
	 */
	public static final int PARAM_SIZE = 4; // sizeof(float)

	private final ByteBuffer commands;
	private final ByteBuffer params;

	/**
	 * Creates a decoder that will decode the path encoded in two byte arrays.
	 *
	 * @param commands
	 *            the encoded path commands (the first available data is the first path's command)
	 * @param params
	 *            the encoded commands' attributes (the first available data is the first command's attribute)
	 */
	public TwoArraysPathData(ByteBuffer commands, ByteBuffer params) {
		this.commands = commands;
		this.params = params;
	}

	@Override
	public boolean hasRemainingCommand() {
		return this.commands.hasRemaining();
	}

	@Override
	public int nextCommand() {
		return Byte.toUnsignedInt(this.commands.get());
	}

	@Override
	public double nextCoordinate() {
		return this.params.getFloat();
	}
}
